package com.supwisdom.institute.backend.biz.api.v1.admin;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.supwisdom.institute.backend.biz.api.vo.request.BizCreateRequest;
import com.supwisdom.institute.backend.biz.api.vo.request.BizQueryRequest;
import com.supwisdom.institute.backend.biz.api.vo.request.BizUpdateRequest;
import com.supwisdom.institute.backend.biz.api.vo.response.BizCreateResponseData;
import com.supwisdom.institute.backend.biz.api.vo.response.BizLoadResponseData;
import com.supwisdom.institute.backend.biz.api.vo.response.BizQueryResponseData;
import com.supwisdom.institute.backend.biz.api.vo.response.BizRemoveResponseData;
import com.supwisdom.institute.backend.biz.api.vo.response.BizUpdateResponseData;
import com.supwisdom.institute.backend.biz.domain.entity.Biz;
import com.supwisdom.institute.backend.biz.domain.exception.BizException;
import com.supwisdom.institute.backend.biz.domain.service.BizService;
import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.vo.response.DefaultApiResponse;

@Api(value = "BizAdminBiz", tags = { "BizAdminBiz" }, description = "Biz示例接口")
@RestController
@RequestMapping("/v1/admin/biz")
public class AdminBizController {
  
  @Autowired
  private BizService bizService;

  /**
   * @param bizQueryRequest
   * @return
   */
  @ApiOperation(value = "查询配置列表", notes = "查询配置列表", nickname = "systemAdminBizQuery")
  @ApiImplicitParams({
    @ApiImplicitParam(name = "loadAll", value = "是否加载全部", required = true, dataType = "boolean", paramType = "query", defaultValue = "false"),
    @ApiImplicitParam(name = "pageIndex", value = "分页 - 页码", required = true, dataType = "int", paramType = "query", defaultValue = "0", example = "0"),
    @ApiImplicitParam(name = "pageSize", value = "分页 - 每页记录数", required = true, dataType = "int", paramType = "query", defaultValue = "20", example = "20"),
    @ApiImplicitParam(name = "mapBean[deleted]", value = "查询条件 - 删除状态 (精确)", required = false, dataType = "boolean", paramType = "query"),
    @ApiImplicitParam(name = "mapBean[categoryCode]", value = "查询条件 - 分类代码 (精确)", required = false, dataType = "string", paramType = "query"),
    @ApiImplicitParam(name = "mapBean[categoryName]", value = "查询条件 - 分类名称 (模糊)", required = false, dataType = "string", paramType = "query"),
    @ApiImplicitParam(name = "mapBean[name]", value = "查询条件 - 名称 (模糊)", required = false, dataType = "string", paramType = "query"),
    @ApiImplicitParam(name = "mapBean[description]", value = "查询条件 - 描述 (模糊)", required = false, dataType = "string", paramType = "query"),
    @ApiImplicitParam(name = "mapBean[bizKey]", value = "查询条件 - 配置Key (精确)", required = false, dataType = "string", paramType = "query"),
  })
  @RequestMapping(method = RequestMethod.GET, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<BizQueryResponseData> query(BizQueryRequest bizQueryRequest) {
    
    Page<Biz> page = bizService.selectPageList(
        bizQueryRequest.isLoadAll(),
        bizQueryRequest.getPageIndex(),
        bizQueryRequest.getPageSize(),
        bizQueryRequest.getMapBean(),
        bizQueryRequest.getOrderBy());

    BizQueryResponseData resp = BizQueryResponseData.of(bizQueryRequest).build(page);
    
    return new DefaultApiResponse<BizQueryResponseData>(resp);
  }

  /**
   * @param id
   * @return
   */
  @ApiOperation(value = "根据ID获取配置项", notes = "根据ID获取配置项", nickname="systemAdminBizLoad")
  @RequestMapping(method = RequestMethod.GET, path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<BizLoadResponseData> load(
      @PathVariable("id") String id) {
    
    if (id == null || id.length() == 0) {
      throw BizException.newInstance("exception.get.id.must.not.empty");
    }
    
    Biz biz = bizService.selectById(id);
    
    if (biz == null) {
      throw BizException.newInstance("exception.get.domain.not.exist");
    }

    BizLoadResponseData resp = BizLoadResponseData.of(biz);
    
    return new DefaultApiResponse<BizLoadResponseData>(resp);
  }
  
  /**
   * @param bizCreateRequest
   * @return
   */
  @ApiOperation(value = "创建配置项", notes = "创建配置项", nickname = "systemAdminBizCreate")
  @RequestMapping(method = RequestMethod.POST, produces = MimeTypeUtils.APPLICATION_JSON_VALUE, consumes = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.CREATED)
  @ResponseBody
  public DefaultApiResponse<BizCreateResponseData> create(
      @RequestBody BizCreateRequest bizCreateRequest) {

    // FIXME: 验证数据有效性

    Biz entity = EntityUtils.copy(bizCreateRequest, new Biz());

    Biz ret = bizService.insert(entity);

    BizCreateResponseData resp = BizCreateResponseData.of(ret);

    return new DefaultApiResponse<BizCreateResponseData>(resp);
  }

  @ApiOperation(value = "更新", notes = "更新", nickname = "systemAdminBizUpdate")
  @RequestMapping(method = RequestMethod.PUT, path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE, consumes = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<BizUpdateResponseData> update(
      @PathVariable("id") String id,
      @RequestBody BizUpdateRequest bizUpdateRequest) {

    if (id == null || id.length() == 0) {
      throw BizException.newInstance("exception.update.id.must.not.empty");
    }

    Biz tmp = bizService.selectById(id);
    if (tmp == null) {
      throw BizException.newInstance("exception.update.domain.not.exist");
    }

    Biz entity = EntityUtils.copy(bizUpdateRequest, new Biz());
    entity.setId(id);

    entity = EntityUtils.merge(tmp, entity);
    
    Biz ret = bizService.update(entity);

    BizUpdateResponseData resp = BizUpdateResponseData.of(ret);
    
    return new DefaultApiResponse<BizUpdateResponseData>(resp);
  }
  
  @ApiOperation(value = "删除", notes = "删除", nickname = "systemAdminBizRemove")
  @RequestMapping(method = RequestMethod.DELETE, path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<BizRemoveResponseData> remove(
      @PathVariable("id") String id) {
    
    if (id == null || id.length() == 0) {
      throw BizException.newInstance("exception.remove.id.must.not.empty");
    }

    Biz tmp = bizService.selectById(id);
    if (tmp == null) {
      throw BizException.newInstance("exception.remove.domain.not.exist");
    }
    
    bizService.deleteById(id);
    
    BizRemoveResponseData resp = BizRemoveResponseData.of(tmp);
    
    return new DefaultApiResponse<BizRemoveResponseData>(resp);
  }

}
