package com.supwisdom.institute.backend.base.api.v1.controller.admin;

import java.util.List;

import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.ApplicationCreateRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.ApplicationDeleteBatchRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.ApplicationQueryRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.ApplicationUpdateRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.ApplicationCreateResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.ApplicationDeleteBatchResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.ApplicationLoadResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.ApplicationQueryResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.ApplicationRemoveResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.ApplicationUpdateResponseData;
import com.supwisdom.institute.backend.base.domain.entity.Application;
import com.supwisdom.institute.backend.base.domain.service.ApplicationService;
import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.vo.response.DefaultApiResponse;

@Api(value = "BaseAdminApplication", tags = { "BaseAdminApplication" }, description = "应用的操作接口")
@Slf4j
@RestController
@RequestMapping("/v1/admin/applications")
public class AdminApplicationController {

  @Autowired
  private ApplicationService applicationService;
  

  @GetMapping(produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ApplicationQueryResponseData> query(ApplicationQueryRequest queryRequest) {

    Page<Application> page = applicationService.selectPageList(
        queryRequest.isLoadAll(), 
        queryRequest.getPageIndex(), 
        queryRequest.getPageSize(),
        queryRequest.getMapBean(),
        queryRequest.getOrderBy());

    ApplicationQueryResponseData data = ApplicationQueryResponseData.of(queryRequest).build(page);

    return new DefaultApiResponse<ApplicationQueryResponseData>(data);
  }
  
  @GetMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ApplicationLoadResponseData> load(@PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Application application = applicationService.selectById(id);

    if (application == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }
    
    ApplicationLoadResponseData data = ApplicationLoadResponseData.of(application);

    return new DefaultApiResponse<ApplicationLoadResponseData>(data);
  }

  @PostMapping(consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ApplicationCreateResponseData> create(
      @RequestBody ApplicationCreateRequest createRequest) {
    
    // FIXME: 验证数据有效性
    
    Application application = createRequest.getEntity();
    
    Application ret = applicationService.insert(application);
    
    ApplicationCreateResponseData data = ApplicationCreateResponseData.build(ret);

    return new DefaultApiResponse<ApplicationCreateResponseData>(data);
  }
  
  @PutMapping(path = "/{id}", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ApplicationUpdateResponseData> update(
      @PathVariable("id") String id, 
      @RequestBody ApplicationUpdateRequest updateRequest) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.update.id.must.not.empty");
    }

    Application tmp = applicationService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.update.domain.not.exist");
    }

    Application application = updateRequest.getEntity();
    application.setId(id);

    application = EntityUtils.merge(tmp, application);

    Application ret = applicationService.update(application);

    ApplicationUpdateResponseData data = ApplicationUpdateResponseData.build(ret);
    
    return new DefaultApiResponse<ApplicationUpdateResponseData>(data);
  }

  @DeleteMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ApplicationRemoveResponseData> delete(
      @PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.delete.id.must.not.empty"); // FIXME: RestException
    }

    Application tmp = applicationService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.delete.domain.not.exist"); // FIXME: RestException
    }

    applicationService.deleteById(id);

    ApplicationRemoveResponseData data = ApplicationRemoveResponseData.build(tmp);
    return new DefaultApiResponse<ApplicationRemoveResponseData>(data);
  }

  @DeleteMapping(path = "/batch", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ApplicationDeleteBatchResponseData> deleteBatch(
      @RequestBody ApplicationDeleteBatchRequest deleteBatchRequest) {
    
    List<String> ids = deleteBatchRequest.getIds();
    
    applicationService.deleteBatch(ids);
    
    ApplicationDeleteBatchResponseData data = ApplicationDeleteBatchResponseData.build(ids);
    return new DefaultApiResponse<ApplicationDeleteBatchResponseData>(data);
  }

}
