package com.supwisdom.institute.backend.base.api.v1.controller.admin;

import java.util.HashMap;

import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.GroupCreateRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.GroupQueryRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.GroupRelateAccountsRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.GroupRelateRolesRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.GroupRelatedAccountsRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.GroupRelatedRolesRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.request.GroupUpdateRequest;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.GroupCreateResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.GroupLoadResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.GroupQueryResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.GroupRelateAccountsResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.GroupRelateRolesResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.GroupRelatedAccountsResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.GroupRelatedRolesResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.GroupRemoveResponseData;
import com.supwisdom.institute.backend.base.api.v1.vo.admin.response.GroupUpdateResponseData;
import com.supwisdom.institute.backend.base.domain.entity.AccountGroup;
import com.supwisdom.institute.backend.base.domain.entity.Group;
import com.supwisdom.institute.backend.base.domain.entity.GroupRole;
import com.supwisdom.institute.backend.base.domain.service.GroupService;
import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.vo.response.DefaultApiResponse;

@Api(value = "BaseAdminGroup", tags = { "BaseAdminGroup" }, description = "用户组的操作接口")
@Slf4j
@RestController
@RequestMapping("/v1/admin/groups")
public class AdminGroupController {

  @Autowired
  private GroupService groupService;

  @GetMapping(produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<GroupQueryResponseData> query(GroupQueryRequest queryRequest) {

    Page<Group> page = groupService.selectPageList(
        queryRequest.isLoadAll(), 
        queryRequest.getPageIndex(), 
        queryRequest.getPageSize(),
        queryRequest.getMapBean(),
        queryRequest.getOrderBy());

    GroupQueryResponseData data = GroupQueryResponseData.of(queryRequest).build(page);

    return new DefaultApiResponse<GroupQueryResponseData>(data);
  }
  
  @GetMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<GroupLoadResponseData> load(@PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Group group = groupService.selectById(id);

    if (group == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }
    
    GroupLoadResponseData data = GroupLoadResponseData.of(group);

    return new DefaultApiResponse<GroupLoadResponseData>(data);
  }

  @PostMapping(consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<GroupCreateResponseData> create(
      @RequestBody GroupCreateRequest createRequest) {
    
    // FIXME: 验证数据有效性
    
    Group group = createRequest.getEntity();
    
    Group ret = groupService.insert(group);
    
    GroupCreateResponseData data = GroupCreateResponseData.build(ret);

    return new DefaultApiResponse<GroupCreateResponseData>(data);
  }
  
  @PutMapping(path = "/{id}", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<GroupUpdateResponseData> update(
      @PathVariable("id") String id, 
      @RequestBody GroupUpdateRequest updateRequest) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.update.id.must.not.empty");
    }

    Group tmp = groupService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.update.domain.not.exist");
    }

    Group group = updateRequest.getEntity();
    group.setId(id);

    group = EntityUtils.merge(tmp, group);

    Group ret = groupService.update(group);

    GroupUpdateResponseData data = GroupUpdateResponseData.build(ret);
    
    return new DefaultApiResponse<GroupUpdateResponseData>(data);
  }

  @DeleteMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<GroupRemoveResponseData> delete(
      @PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.delete.id.must.not.empty"); // FIXME: RestException
    }

    Group tmp = groupService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.delete.domain.not.exist"); // FIXME: RestException
    }

    groupService.deleteById(id);

    GroupRemoveResponseData data = GroupRemoveResponseData.build(tmp);
    return new DefaultApiResponse<GroupRemoveResponseData>(data);
  }

  
  
  @RequestMapping(method = RequestMethod.GET, path = "/{id}/accounts", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<GroupRelatedAccountsResponseData> groupAccounts(
      @PathVariable("id") String id, 
      GroupRelatedAccountsRequest request) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Group group = groupService.selectById(id);

    if (group == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    if (request.getMapBean() == null) {
      request.setMapBean(new HashMap<String, Object>());
    }
    request.getMapBean().put("groupId", group.getId());

    Page<AccountGroup> page = groupService.selectGroupAccounts(
        request.getPageIndex(),
        request.getPageSize(), 
        request.getMapBean());

    GroupRelatedAccountsResponseData data = GroupRelatedAccountsResponseData.of(request).build(page);

    return new DefaultApiResponse<GroupRelatedAccountsResponseData>(data);
  }

  @RequestMapping(method = RequestMethod.POST, path = "/{id}/accounts", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<GroupRelateAccountsResponseData> relateAccounts(
      @PathVariable("id") String id, 
      @RequestBody GroupRelateAccountsRequest groupAccounts) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Group group = groupService.selectById(id);

    if (group == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    groupService.relateGroupAccounts(group, groupAccounts.getGroupAccounts());

    GroupRelateAccountsResponseData data = GroupRelateAccountsResponseData.of("info.relate.success");

    return new DefaultApiResponse<GroupRelateAccountsResponseData>(data);
  }

  
  @RequestMapping(method = RequestMethod.GET, path = "/{id}/roles", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<GroupRelatedRolesResponseData> groupRoles(
      @PathVariable("id") String id, 
      GroupRelatedRolesRequest request) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Group group = groupService.selectById(id);

    if (group == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    if (request.getMapBean() == null) {
      request.setMapBean(new HashMap<String, Object>());
    }
    request.getMapBean().put("groupId", group.getId());

    Page<GroupRole> page = groupService.selectGroupRoles(
        request.getPageIndex(),
        request.getPageSize(), 
        request.getMapBean());

    GroupRelatedRolesResponseData data = GroupRelatedRolesResponseData.of(request).build(page);

    return new DefaultApiResponse<GroupRelatedRolesResponseData>(data);
  }

  @RequestMapping(method = RequestMethod.POST, path = "/{id}/roles", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<GroupRelateRolesResponseData> relateRoles(
      @PathVariable("id") String id, 
      @RequestBody GroupRelateRolesRequest groupRoles) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Group group = groupService.selectById(id);

    if (group == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    groupService.relateGroupRoles(group, groupRoles.getGroupRoles());

    GroupRelateRolesResponseData data = GroupRelateRolesResponseData.of("info.relate.success");

    return new DefaultApiResponse<GroupRelateRolesResponseData>(data);
  }

}
