package com.supwisdom.institute.backend.common.core.distributedlock.aop;

import java.lang.reflect.Method;
import java.util.Arrays;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;

import com.supwisdom.institute.backend.common.core.distributedlock.DistributedLockAutoConfiguration;
import com.supwisdom.institute.backend.common.core.distributedlock.DistributedLockHandler;
import com.supwisdom.institute.backend.common.core.distributedlock.aop.DistributedLock.LockFailAction;

@Aspect
@Configuration
@ConditionalOnClass({ProceedingJoinPoint.class, DistributedLockHandler.class})
@AutoConfigureAfter(DistributedLockAutoConfiguration.class)
public class DistributedLockAspectConfiguration {
  
  private final Logger logger = LoggerFactory.getLogger(DistributedLockAspectConfiguration.class);
  
  @Autowired
  private DistributedLockHandler distributedLockHandler;

  @Pointcut("@annotation(com.supwisdom.institute.backend.common.core.distributedlock.aop.DistributedLock)")
  private void lockPoint(){
    
  }
  
  @Around("lockPoint()")
  public Object around(ProceedingJoinPoint pjp) throws Throwable{
    Method method = ((MethodSignature) pjp.getSignature()).getMethod();
    DistributedLock distributedLockAnnotation = method.getAnnotation(DistributedLock.class);
    String key = distributedLockAnnotation.value();
    if(StringUtils.isEmpty(key)){
      Object[] args = pjp.getArgs();
      key = Arrays.toString(args);
    }
    int retryTimes = distributedLockAnnotation.action().equals(LockFailAction.CONTINUE) ? distributedLockAnnotation.retryTimes() : 0;
    boolean lock = distributedLockHandler.lock(key, distributedLockAnnotation.keepMills(), retryTimes, distributedLockAnnotation.sleepMills());
    if(!lock) {
      logger.debug("get lock failed : " + key);
      return null;
    }
    
    //得到锁,执行方法，释放锁
    logger.debug("get lock success : " + key);
    try {
      return pjp.proceed();
    } catch (Exception e) {
      logger.error("execute locked method occured an exception", e);
    } finally {
      boolean releaseResult = distributedLockHandler.releaseLock(key);
      logger.debug("release lock : " + key + (releaseResult ? " success" : " failed"));
    }
    return null;
  }
}