package com.supwisdom.institute.backend.common.util;

import lombok.extern.slf4j.Slf4j;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.util.Assert;

import java.lang.reflect.*;
import java.util.Date;

/**
 * 利用反射进行操作的一个工具类
 *
 * @author fengpy
 */
@SuppressWarnings("rawtypes")
@Slf4j
public class ReflectUtils {

  private static final String SETTER_PREFIX = "set";

  private static final String GETTER_PREFIX = "get";

  private static final String CGLIB_CLASS_SEPARATOR = "$$";

  /**
   * 利用反射获取指定对象里面的指定属性
   *
   * @param obj
   *          目标对象
   * @param fieldName
   *          目标属性
   * @return 目标字段
   */
  private static Field getField(Object obj, String fieldName) {
    Field field = null;
    for (Class<?> clazz = obj.getClass(); clazz != Object.class; clazz = clazz.getSuperclass()) {
      try {
        field = clazz.getDeclaredField(fieldName);
        break;
      } catch (NoSuchFieldException e) {
        // 这里不用做处理，子类没有该字段可能对应的父类有，都没有就返回null。
      }
    }
    return field;
  }

  /**
   * 调用Getter方法.
   * 支持多级，如：对象名.对象名.方法
   */
  public static Object invokeGetter(Object obj, String propertyName) {
    Object object = obj;
    for (String name : StringUtils.split(propertyName, ".")){
      String getterMethodName = GETTER_PREFIX + StringUtils.capitalize(name);
      object = invokeMethod(object, getterMethodName, new Class[] {}, new Object[] {});
    }
    return object;
  }

  /**
   * 调用Setter方法, 仅匹配方法名。
   * 支持多级，如：对象名.对象名.方法
   */
  public static void invokeSetter(Object obj, String propertyName, Object value, Class valueCla) throws IllegalAccessException, InstantiationException {
    Object object = obj;
    String[] names = StringUtils.split(propertyName, ".");
    for (int i=0; i<names.length; i++){
      if(i<names.length-1){
        String getterMethodName = GETTER_PREFIX + StringUtils.capitalize(names[i]);
        object = invokeMethod(object, getterMethodName, new Class[] {}, new Object[] {});
      }else{
        String setterMethodName = SETTER_PREFIX + StringUtils.capitalize(names[i]);
        value = doSpecial(value,valueCla);
        invokeMethodByName(object, setterMethodName, new Object[] { value });
      }
    }
  }

  /**
   * 直接读取对象属性值, 无视private/protected修饰符, 不经过getter函数.
   */
  public static Object getFieldValue(final Object obj, final String fieldName) {
    Field field = getAccessibleField(obj, fieldName);

    if (field == null) {
      throw new IllegalArgumentException("Could not find field [" + fieldName + "] on target [" + obj + "]");
    }

    Object result = null;
    try {
      result = field.get(obj);
    } catch (IllegalAccessException e) {
      log.error("不可能抛出的异常{}", e.getMessage());
    }
    return result;
  }

  /**
   * 直接设置对象属性值, 无视private/protected修饰符, 不经过setter函数.
   */
  public static void setFieldValue(final Object obj, final String fieldName, final Object value, Class valueCla) {
    Field field = getAccessibleField(obj, fieldName);

    if (field == null) {
      throw new IllegalArgumentException("Could not find field [" + fieldName + "] on target [" + obj + "]");
    }

    try {
      doSpecial(value,valueCla);
      field.set(obj, value);
    } catch (IllegalAccessException e) {
      log.error("不可能抛出的异常:{}", e.getMessage());
    }
  }

  /**
   * 直接调用对象方法, 无视private/protected修饰符.
   * 用于一次性调用的情况，否则应使用getAccessibleMethod()函数获得Method后反复调用.
   * 同时匹配方法名+参数类型，
   */
  public static Object invokeMethod(final Object obj, final String methodName, final Class<?>[] parameterTypes,
                                    final Object[] args) {
    Method method = getAccessibleMethod(obj, methodName, parameterTypes);
    if (method == null) {
      throw new IllegalArgumentException("Could not find method [" + methodName + "] on target [" + obj + "]");
    }

    try {
      return method.invoke(obj, args);
    } catch (Exception e) {
      throw convertReflectionExceptionToUnchecked(e);
    }
  }

  /**
   * 直接调用对象方法, 无视private/protected修饰符，
   * 用于一次性调用的情况，否则应使用getAccessibleMethodByName()函数获得Method后反复调用.
   * 只匹配函数名，如果有多个同名函数调用第一个。
   */
  public static Object invokeMethodByName(final Object obj, final String methodName, final Object[] args) {
    Method method = getAccessibleMethodByName(obj, methodName);
    if (method == null) {
      throw new IllegalArgumentException("Could not find method [" + methodName + "] on target [" + obj + "]");
    }

    try {
      return method.invoke(obj, args);
    } catch (Exception e) {
      throw convertReflectionExceptionToUnchecked(e);
    }
  }

  /**
   * 循环向上转型, 获取对象的DeclaredField, 并强制设置为可访问.
   *
   * 如向上转型到Object仍无法找到, 返回null.
   */
  public static Field getAccessibleField(final Object obj, final String fieldName) {
    Validate.notNull(obj, "object can't be null");
    Validate.notBlank(fieldName, "fieldName can't be blank");
    for (Class<?> superClass = obj.getClass(); superClass != Object.class; superClass = superClass.getSuperclass()) {
      try {
        Field field = superClass.getDeclaredField(fieldName);
        makeAccessible(field);
        return field;
      } catch (NoSuchFieldException e) {//NOSONAR
        // Field不在当前类定义,继续向上转型
        continue;// new add
      }
    }
    return null;
  }

  /**
   * 循环向上转型, 获取对象的DeclaredMethod,并强制设置为可访问.
   * 如向上转型到Object仍无法找到, 返回null.
   * 匹配函数名+参数类型。
   *
   * 用于方法需要被多次调用的情况. 先使用本函数先取得Method,然后调用Method.invoke(Object obj, Object... args)
   */
  public static Method getAccessibleMethod(final Object obj, final String methodName,
                                           final Class<?>... parameterTypes) {
    Validate.notNull(obj, "object can't be null");
    Validate.notBlank(methodName, "methodName can't be blank");

    for (Class<?> searchType = obj.getClass(); searchType != Object.class; searchType = searchType.getSuperclass()) {
      try {
        Method method = searchType.getDeclaredMethod(methodName, parameterTypes);
        makeAccessible(method);
        return method;
      } catch (NoSuchMethodException e) {
        // Method不在当前类定义,继续向上转型
        continue;// new add
      }
    }
    return null;
  }

  /**
   * 循环向上转型, 获取对象的DeclaredMethod,并强制设置为可访问.
   * 如向上转型到Object仍无法找到, 返回null.
   * 只匹配函数名。
   *
   * 用于方法需要被多次调用的情况. 先使用本函数先取得Method,然后调用Method.invoke(Object obj, Object... args)
   */
  public static Method getAccessibleMethodByName(final Object obj, final String methodName) {
    Validate.notNull(obj, "object can't be null");
    Validate.notBlank(methodName, "methodName can't be blank");

    for (Class<?> searchType = obj.getClass(); searchType != Object.class; searchType = searchType.getSuperclass()) {
      Method[] methods = searchType.getDeclaredMethods();
      for (Method method : methods) {
        if (method.getName().equals(methodName)) {
          makeAccessible(method);
          return method;
        }
      }
    }
    return null;
  }

  /**
   * 改变private/protected的方法为public，尽量不调用实际改动的语句，避免JDK的SecurityManager抱怨。
   */
  public static void makeAccessible(Method method) {
    if ((!Modifier.isPublic(method.getModifiers()) || !Modifier.isPublic(method.getDeclaringClass().getModifiers()))
            && !method.isAccessible()) {
      method.setAccessible(true);
    }
  }

  /**
   * 改变private/protected的成员变量为public，尽量不调用实际改动的语句，避免JDK的SecurityManager抱怨。
   */
  public static void makeAccessible(Field field) {
    if ((!Modifier.isPublic(field.getModifiers()) || !Modifier.isPublic(field.getDeclaringClass().getModifiers()) || Modifier
            .isFinal(field.getModifiers())) && !field.isAccessible()) {
      field.setAccessible(true);
    }
  }

  /**
   * 通过反射, 获得Class定义中声明的泛型参数的类型, 注意泛型必须定义在父类处
   * 如无法找到, 返回Object.class.
   * eg.
   * public UserDao extends HibernateDao<User>
   *
   * @param clazz The class to introspect
   * @return the first generic declaration, or Object.class if cannot be determined
   */
  @SuppressWarnings("unchecked")
  public static <T> Class<T> getClassGenricType(final Class clazz) {
    return getClassGenricType(clazz, 0);
  }

  /**
   * 通过反射, 获得Class定义中声明的父类的泛型参数的类型.
   * 如无法找到, 返回Object.class.
   *
   * 如public UserDao extends HibernateDao<User,Long>
   *
   * @param clazz clazz The class to introspect
   * @param index the Index of the generic ddeclaration,start from 0.
   * @return the index generic declaration, or Object.class if cannot be determined
   */
  public static Class getClassGenricType(final Class clazz, final int index) {

    Type genType = clazz.getGenericSuperclass();

    if (!(genType instanceof ParameterizedType)) {
      log.warn(clazz.getSimpleName() + "'s superclass not ParameterizedType");
      return Object.class;
    }

    Type[] params = ((ParameterizedType) genType).getActualTypeArguments();

    if (index >= params.length || index < 0) {
      log.warn("Index: " + index + ", Size of " + clazz.getSimpleName() + "'s Parameterized Type: "
              + params.length);
      return Object.class;
    }
    if (!(params[index] instanceof Class)) {
      log.warn(clazz.getSimpleName() + " not set the actual class on superclass generic parameter");
      return Object.class;
    }

    return (Class) params[index];
  }

  public static Class<?> getUserClass(Object instance) {
    Assert.notNull(instance, "Instance must not be null");
    Class clazz = instance.getClass();
    if (clazz != null && clazz.getName().contains(CGLIB_CLASS_SEPARATOR)) {
      Class<?> superClass = clazz.getSuperclass();
      if (superClass != null && !Object.class.equals(superClass)) {
        return superClass;
      }
    }
    return clazz;

  }

  /**
   * 将反射时的checked exception转换为unchecked exception.
   */
  public static RuntimeException convertReflectionExceptionToUnchecked(Exception e) {
    if (e instanceof IllegalAccessException || e instanceof IllegalArgumentException
            || e instanceof NoSuchMethodException) {
      return new IllegalArgumentException(e);
    } else if (e instanceof InvocationTargetException) {
      return new RuntimeException(((InvocationTargetException) e).getTargetException());
    } else if (e instanceof RuntimeException) {
      return (RuntimeException) e;
    }
    return new RuntimeException("Unexpected Checked Exception.", e);
  }

  /**
   * 类型转换
   *
   * @param clazz
   *          ：目标类型
   * @param source
   *          ：待转换对象
   * @return ：目标对象
   */
  public static Object typeConversion(Class<?> clazz, String source) {

    if (clazz == null) {
      throw new IllegalArgumentException("clazz should not be null");
    }

    Object targetObj = null;
    String nameType = clazz.getName();

    if ("java.lang.Integer".equals(nameType) || "int".equals(nameType)) {
      targetObj = Integer.valueOf(source);
    } else if ("java.lang.String".equals(nameType) || "string".equals(nameType)) {
      targetObj = source;
    } else if ("java.lang.Float".equals(nameType) || "float".equals(nameType)) {
      targetObj = Float.valueOf(source);
    } else if ("java.lang.Double".equals(nameType) || "double".equals(nameType)) {
      targetObj = Double.valueOf(source);
    } else if ("java.lang.Boolean".equals(nameType) || "boolean".equals(nameType)) {
      targetObj = Boolean.valueOf(source);
    } else if ("java.lang.Long".equals(nameType) || "long".equals(nameType)) {
      targetObj = Long.valueOf(source);
    } else if ("java.lang.Short".equals(nameType) || "short".equals(nameType)) {
      targetObj = Short.valueOf(source);
    } else if ("java.lang.Character".equals(nameType) || "char".equals(nameType)) {
      targetObj = source.charAt(1);
    }else if ("java.util.Date".equals(nameType)) {
      targetObj = new Date(source);
    }

    return targetObj;
  }


  /**
   * 根据类的全路径获取class
   * @param fullPath
   *          ：类的全路径
   * @return ：class
   */
  public static Class fullPath2Class(String fullPath) {
    Class cl = null;
    try {
      ClassLoader loader = ClassLoader.getSystemClassLoader();
      cl = loader.loadClass(fullPath);
    } catch (ClassNotFoundException e) {
      e.printStackTrace();
    }
    return cl;
  }

    private static Object doSpecial(Object value, Class valueCla){
        //TODO 针对sql脚本的类型和实体entity的类型不一致做的特殊处理(有待优化)
        if(valueCla.equals(Boolean.class)&&value.getClass().equals(Integer.class)){
            value = typeConversion(Boolean.class,value.toString());
        }
        return value;
    }
}
