package com.supwisdom.infras.security.reactive.jwt;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.authentication.ReactiveAuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.ReactiveUserDetailsService;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.RestController;

import reactor.core.publisher.Mono;

import com.supwisdom.infras.security.configure.jwt.util.SignUtil;
import com.supwisdom.infras.security.core.userdetails.InfrasUser;
import com.supwisdom.infras.security.utils.JWTTokenUtil;

@RestController
public class JWTTokenController {
  
  private static final Logger logger = LoggerFactory.getLogger(JWTTokenController.class);
  
  @Value("${infras.security.jwt.token.sign.enabled:false}")
  private boolean signEnabled;
  @Value("${infras.security.jwt.token.sign.key:}")
  private String signKey;

  @Value("${infras.security.jwt.token.authorization.prefix:Bearer}")
  private String authorizationPrefix;

  @Autowired
  private ReactiveAuthenticationManager reactiveAuthenticationManager;
  @Autowired
  private ReactiveUserDetailsService reactiveUserDetailsService;
  @Autowired
  private JWTTokenUtil jwtTokenUtil;
  
  /**
   * 用户登录
   * 
   * curl -i -s -X POST 'http://localhost:8080/jwt/token/login' -d 'username=user&password=user' -H 'Content-Type: application/json'
   *
   * @param username 用户名
   * @param password 密码
   * @return 操作结果
   * @throws AuthenticationException 错误信息
   */
  @PostMapping(path = "/jwt/token/login", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE)
  public Mono<String> login(
      @RequestBody LoginRequest loginRequest
      ) throws AuthenticationException {

    return Mono.just(loginRequest).flatMap(req -> {
      try {
        String username = req.getUsername();
        String password = req.getPassword();
        
        // TODO：判断用户的登录频次
        
        UsernamePasswordAuthenticationToken upToken = new UsernamePasswordAuthenticationToken(username, password);
        
        return reactiveAuthenticationManager.authenticate(upToken);
      } catch (AuthenticationException e) {
        // TODO: 记录失败
        throw e;
      }
    }).filter(a -> a.isAuthenticated() && a.getPrincipal() instanceof InfrasUser)
    .map(Authentication::getPrincipal)
    .cast(InfrasUser.class)
    .map(userDetails -> {
      String jwtToken = generateToken(userDetails);
      return jwtToken;
    })
    ;
    
    // 判断签名
//    if (signEnabled) {
//      String timestamp = request.queryParam("timestamp").get();
//      String sign = request.queryParam("sign").get();
//      
//      if (timestamp == null || timestamp.isEmpty()) {
//        throw new BadCredentialsException("timestamp error");
//      }
//      
//      if (sign == null || sign.isEmpty()) {
//        throw new BadCredentialsException("sign error");
//      }
//      
//      String data = timestamp + username + password;
//      
//      if (!SignUtil.checkIntervalTime(timestamp, 10000)) {
//        throw new BadCredentialsException("timestamp error");
//      }
//      
//      String currentSign = SignUtil.HMACSHA1(data, signKey);
//      if (!currentSign.equals(sign)) {
//        throw new BadCredentialsException("sign error");
//      }
//    }
    
//    String username = loginRequest.getUsername();
//    String password = loginRequest.getPassword();
//
//    // TODO：判断用户的登录频次
//    
//    try {
//      UsernamePasswordAuthenticationToken upToken = new UsernamePasswordAuthenticationToken(username, password);
//      
//      Mono<String> token = 
//      reactiveAuthenticationManager.authenticate(upToken)
//        .filter(a -> a.isAuthenticated() && a.getPrincipal() instanceof UserDetails)
//        .map(Authentication::getPrincipal)
//        .cast(UserDetails.class)
//        .map(userDetails -> {
//          String jwtToken = generateToken(userDetails);
//          return jwtToken;
//        });
//
////      reactiveAuthenticationManager.authenticate(upToken)
////        .filter(a -> a.isAuthenticated() && a.getPrincipal() instanceof String)
////        .map(Authentication::getPrincipal)
////        .cast(String.class)
////        .map(principal -> {
////          return reactiveUserDetailsService.findByUsername(principal)
////            .map(userDetails -> {
////              String jwtToken = generateToken(userDetails);
////              return jwtToken;
////            });
////        });
//      
//      return token.block();
//      
////      reactiveAuthenticationManager.authenticate(upToken)
////        .filter(a -> a.isAuthenticated() && a.getPrincipal() instanceof UserDetails)
////        .map(Authentication::getPrincipal)
////        .cast(UserDetails.class)
////        .map(userDetails -> {
////          String jwtToken = generateToken(userDetails);
////          return jwtToken;
////        })
////      ;
//
//    } catch (AuthenticationException e) {
//      // TODO: 记录失败
//      throw e;
//    }
//
//    //throw new UsernameNotFoundException(String.format("%s not found", username));
  }
  

  /**
   * 生成令牌
   *
   * @param userDetails 用户
   * @return 令牌
   */
  private String generateToken(InfrasUser userDetails) {
    String roles = "";
    for (GrantedAuthority ga : userDetails.getAuthorities()) {
      String role = ga.getAuthority();
      roles += (roles.length() > 0 ? "," : "") + role;
    }
    
    Map<String, Object> claims = new HashMap<>();
    claims.put("sub", userDetails.getUsername());
    claims.put("roles", roles);
    claims.put("created", new Date());
    return jwtTokenUtil.generateToken(claims);
  }

  
  /**
   * 刷新密钥
   * 
   * curl -i -s -X GET 'http://localhost:8080/jwt/token/refreshToken' -H 'Authorization: Bearer JWTToken'
   *
   * @param authorization 原密钥
   * @return 新密钥
   * @throws AuthenticationException 错误信息
   */
  @GetMapping(path = "/jwt/token/refreshToken")
  public String refreshToken(@RequestHeader("Authorization") String authorization) throws AuthenticationException {
    
    if (authorization == null || !authorization.toLowerCase().startsWith(authorizationPrefix.toLowerCase())) {
      return "authorization error";
    }
    
    String token = authorization.substring(authorizationPrefix.length() + 1);
    //if (!jwtTokenUtil.isTokenExpired(token)) {
        return jwtTokenUtil.refreshToken(token);
    //}
    
    //return "error";
  }
  
  /**
   * 注销密钥
   * 
   * curl -i -s -X GET 'http://localhost:8080/jwt/token/logout' -H 'Authorization: Bearer JWTToken'
   * 
   * @param authorization 原密钥
   * @return
   * @throws AuthenticationException
   */
  @GetMapping(path = "/jwt/token/logout")
  public String expireToken(@RequestHeader("Authorization") String authorization) throws AuthenticationException {
    
    if (authorization == null || !authorization.toLowerCase().startsWith(authorizationPrefix.toLowerCase())) {
      return "authorization error";
    }
    
    String token = authorization.substring(authorizationPrefix.length() + 1);
    
    jwtTokenUtil.expireToken(token);
    
    return "success";
  }
  
}
