package com.supwisdom.institute.backend.system.api.v1.admin;

import java.util.HashMap;
import java.util.List;

import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.vo.response.DefaultApiResponse;
import com.supwisdom.institute.backend.system.api.vo.request.AccountCreateRequest;
import com.supwisdom.institute.backend.system.api.vo.request.AccountDeleteBatchRequest;
import com.supwisdom.institute.backend.system.api.vo.request.AccountQueryRequest;
import com.supwisdom.institute.backend.system.api.vo.request.AccountRelateGroupsRequest;
import com.supwisdom.institute.backend.system.api.vo.request.AccountRelateRolesRequest;
import com.supwisdom.institute.backend.system.api.vo.request.AccountRelatedGroupsRequest;
import com.supwisdom.institute.backend.system.api.vo.request.AccountRelatedRolesRequest;
import com.supwisdom.institute.backend.system.api.vo.request.AccountUpdateRequest;
import com.supwisdom.institute.backend.system.api.vo.response.AccountCreateResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.AccountDeleteBatchResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.AccountLoadResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.AccountQueryResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.AccountRelateGroupsResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.AccountRelateRolesResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.AccountRelatedGroupsResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.AccountRelatedRolesResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.AccountRemoveResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.AccountUpdateResponseData;
import com.supwisdom.institute.backend.system.domain.entity.Account;
import com.supwisdom.institute.backend.system.domain.entity.AccountGroup;
import com.supwisdom.institute.backend.system.domain.entity.AccountRole;
import com.supwisdom.institute.backend.system.domain.service.AccountService;

@Api(value = "SystemAdminAccount", tags = { "SystemAdminAccount" }, description = "帐号的操作接口")
@Slf4j
@RestController
@RequestMapping("/v1/admin/accounts")
public class AdminAccountController {
  
  @Autowired
  private AccountService accountService;

  /**
   * 
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts'
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts?pageIndex=2&pageSize=50'
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts?pageIndex=0&pageSize=20&mapBean[username]=username&mapBean[name]=name&mapBean[status]=1'
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts?pageIndex=0&pageSize=20&mapBean[username]=username&mapBean[name]=name&mapBean[status]=0'
   * 
   * response success:
   * 
   * <pre>
   * {
   *   "pageIndex":0,
   *   "pageSize":20,
   *   "mapBean":null,
   *   "pageCount":1,
   *   "recordCount":1,
   *   "items":[
   *     {
   *       "id":"ff80808164feb8990164feba0de50000",
   *       "companyId":"1",
   *       "deleted":false,
   *       "addAccount":"account","addTime":"2018-08-03T07:39:23.000+0000",
   *       "editAccount":null,"editTime":null,
   *       "deleteAccount":null,"deleteTime":null,
   *       "accountname":"test001",
   *       "password":"test001",
   *       "enabled":true,
   *       "accountNonExpired":true,
   *       "accountNonLocked":true,
   *       "credentialsNonExpired":true,
   *       "name":"测试001",
   *       "status":"1",
   *       "mobile":null,
   *       "email":null
   *     }
   *   ]
   * }
   * </pre>
   * 
   * response error 401:
   * 
   * <pre>
   * {
   *   "timestamp":"2018-08-03T08:48:25.777+0000",
   *   "status":401,
   *   "error":"Http Status 401",
   *   "message":"Unauthorized",
   *   "path":"/api/v1/admin/accounts"
   * }
   * </pre>
   * 
   * @param pagerRequestModel
   * @return
   */
  @GetMapping(produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<AccountQueryResponseData> query(AccountQueryRequest queryRequest) {

    Page<Account> page = accountService.selectPageList(
        queryRequest.isLoadAll(), 
        queryRequest.getPageIndex(), 
        queryRequest.getPageSize(),
        queryRequest.getMapBean(),
        queryRequest.getOrderBy());

    AccountQueryResponseData data = AccountQueryResponseData.of(queryRequest).build(page);

    return new DefaultApiResponse<AccountQueryResponseData>(data);
  }

  /**
   * 
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts/1'
   * 
   * response success:
   * 
   * <pre>
   * {
   *   "id":"ff80808164feb8990164feba0de50000",
   *   "companyId":"1",
   *   "deleted":false,
   *   "addAccount":"account","addTime":"2018-08-03T07:39:23.000+0000",
   *   "editAccount":null,"editTime":null,
   *   "deleteAccount":null,"deleteTime":null,
   *   "username":"test001",
   *   "password":"test001",
   *   "enabled":true,
   *   "accountNonExpired":true,
   *   "accountNonLocked":true,
   *   "credentialsNonExpired":true,
   *   "name":"测试001",
   *   "status":"1",
   *   "mobile":null,
   *   "email":null
   * }
   * </pre>
   * 
   * response error 401:
   * 
   * <pre>
   * {
   *   "timestamp":"2018-08-03T08:43:26.080+0000",
   *   "status":401,
   *   "error":"Http Status 401",
   *   "message":"Unauthorized",
   *   "path":"/api/v1/admin/accounts/ff80808164fecf640164fed269480000"
   * }
   * </pre>
   * 
   * response error 500:
   * 
   * <pre>
   * {
   *   "timestamp":"2018-08-03T07:44:07.963+0000",
   *   "status":500,
   *   "error":"Internal Server Error",
   *   "exception":"java.lang.RuntimeException",
   *   "message":"exception.get.domain.not.exist",
   *   "path":"/api/v1/admin/accounts/1"
   * }
   * </pre>
   * 
   * @param id
   * @return
   */
  @GetMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<AccountLoadResponseData> load(@PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Account account = accountService.selectById(id);

    if (account == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }
    
    AccountLoadResponseData data = AccountLoadResponseData.of(account);

    return new DefaultApiResponse<AccountLoadResponseData>(data);
  }

  /**
   * 
   * curl -i -s -X POST -H 'Content-Type:application/json' -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts' \
   * -d '{"accountname":"test001","password":"test001","enabled":true,"accountNonExpired":true,"accountNonLocked":true,"credentialsNonExpired":true,"name":"测试001","status":"1"}'
   * 
   * response success:
   * 
   * <pre>
   * {
   *   "success":"info.create.success"
   * }
   * </pre>
   * 
   * response error 401:
   * 
   * <pre>
   * {
   *   "timestamp":"2018-08-03T08:48:25.777+0000",
   *   "status":401,
   *   "error":"Http Status 401",
   *   "message":"Unauthorized",
   *   "path":"/api/v1/admin/accounts"
   * }
   * </pre>
   * 
   * response error: // FIXME: save error
   * 
   * <pre>
   * {
   *   "timestamp":"2018-08-03T07:45:43.436+0000",
   *   "status":500,
   *   "error":"Internal Server Error",
   *   "exception":"org.springframework.dao.DataIntegrityViolationException",
   *   "message":"could not execute statement; SQL [n/a]; constraint [null]; nested exception is org.hibernate.exception.ConstraintViolationException: could not execute statement",
   *   "path":"/api/v1/admin/accounts"
   * }
   * </pre>
   * 
   * @param account
   * @return
   */
  @PostMapping(consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<AccountCreateResponseData> create(
      @RequestBody AccountCreateRequest createRequest) {
    
    // FIXME: 验证数据有效性
    
    Account account = createRequest.getEntity();
    
    if (account.getPassword() !=null && account.getPassword().length() > 0 && !account.getPassword().startsWith("{")) {
      //account.setPassword(passwordEncoder.encode(account.getPassword()));
    }

    Account ret = accountService.insert(account);
    
    AccountCreateResponseData data = AccountCreateResponseData.build(ret);

    return new DefaultApiResponse<AccountCreateResponseData>(data);
  }

  /**
   * 
   * curl -i -s -X PUT -H 'Content-Type:application/json' -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts' \
   * -d '{"id":"1","status":"0"}'
   * 
   * response success:
   * 
   * <pre>
   * {
   *   "success":"info.update.success"
   * }
   * </pre>
   * 
   * response error 401:
   * 
   * <pre>
   * {
   *   "timestamp":"2018-08-03T08:48:25.777+0000",
   *   "status":401,
   *   "error":"Http Status 401",
   *   "message":"Unauthorized",
   *   "path":"/api/v1/admin/accounts"
   * }
   * </pre>
   * 
   * curl -i -s -X PUT -H 'Content-Type:application/json' -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts' \
   * -d '{"status":"0"}'
   * 
   * response error:
   * 
   * <pre>
   * {
   *   "timestamp":"2018-08-03T07:50:52.327+0000",
   *   "status":500,
   *   "error":"Internal Server Error",
   *   "exception":"java.lang.RuntimeException",
   *   "message":"exception.update.id.must.not.empty",
   *   "path":"/api/v1/admin/accounts"
   * }
   * </pre>
   * 
   * curl -i -s -X PUT -H 'Content-Type:application/json' -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts' \
   * -d '{"id":"1","status":"0"}'
   * 
   * response error:
   * 
   * <pre>
   * {
   *   "timestamp":"2018-08-03T07:48:24.774+0000",
   *   "status":500,
   *   "error":"Internal Server Error",
   *   "exception":"java.lang.RuntimeException",
   *   "message":"exception.update.domain.not.exist",
   *   "path":"/api/v1/admin/accounts"
   * }
   * </pre>
   * 
   * @param account
   * @return
   */
  @PutMapping(path = "/{id}", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<AccountUpdateResponseData> update(
      @PathVariable("id") String id, 
      @RequestBody AccountUpdateRequest updateRequest) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.update.id.must.not.empty");
    }

    Account tmp = accountService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.update.domain.not.exist");
    }

    Account account = updateRequest.getEntity();
    account.setId(id);

    if (account.getPassword() !=null && account.getPassword().length() > 0 && !account.getPassword().startsWith("{")) {
      //account.setPassword(passwordEncoder.encode(account.getPassword()));
    }

    account = EntityUtils.merge(tmp, account);

    Account ret = accountService.update(account);

    AccountUpdateResponseData data = AccountUpdateResponseData.build(ret);
    
    return new DefaultApiResponse<AccountUpdateResponseData>(data);
  }
  

  /**
   * 
   * curl -i -s -X DELETE -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts/1'
   * 
   * response success:
   * 
   * <pre>
   * {
   *   "success":"info.delete.success"
   * }
   * </pre>
   * 
   * response error 401:
   * 
   * <pre>
   * {
   *   "timestamp":"2018-08-03T08:48:25.777+0000",
   *   "status":401,
   *   "error":"Http Status 401",
   *   "message":"Unauthorized",
   *   "path":"/api/v1/admin/accounts/1"
   * }
   * </pre>
   * 
   * response error 500:
   * 
   * <pre>
   * {
   *   "timestamp":"2018-08-03T08:03:16.364+0000",
   *   "status":500,
   *   "error":"Internal Server Error",
   *   "exception":"java.lang.RuntimeException",
   *   "message":"exception.delete.domain.not.exist",
   *   "path":"/api/v1/admin/accounts/1"
   * }
   * </pre>
   * 
   * @param id
   * @return
   */
  @DeleteMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<AccountRemoveResponseData> delete(
      @PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.delete.id.must.not.empty"); // FIXME: RestException
    }

    Account tmp = accountService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.delete.domain.not.exist"); // FIXME: RestException
    }

    accountService.deleteById(id);

    AccountRemoveResponseData data = AccountRemoveResponseData.build(tmp);
    return new DefaultApiResponse<AccountRemoveResponseData>(data);
  }
  
  @DeleteMapping(path = "/batch", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<AccountDeleteBatchResponseData> deleteBatch(
      @RequestBody AccountDeleteBatchRequest deleteBatchRequest) {
    
    System.out.println(deleteBatchRequest.getIds());
    List<String> ids = deleteBatchRequest.getIds();
    
    accountService.deleteBatch(ids);
    
    AccountDeleteBatchResponseData data = AccountDeleteBatchResponseData.build(ids);
    return new DefaultApiResponse<AccountDeleteBatchResponseData>(data);
  }

  /**
   * 
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts/1/groups'
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts/1/groups?pageIndex=2&pageSize=50'
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts/1/groups?pageIndex=0&pageSize=20&mapBean[groupCode]=groupCode&mapBean[groupName]=groupName'
   * 
   * 
   * 
   * @param id
   * @param pagerRequestModel
   * @return
   */
  @RequestMapping(method = RequestMethod.GET, path = "/{id}/groups", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<AccountRelatedGroupsResponseData> accountGroups(
      @PathVariable("id") String id, 
      AccountRelatedGroupsRequest request) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Account account = accountService.selectById(id);

    if (account == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    if (request.getMapBean() == null) {
      request.setMapBean(new HashMap<String, Object>());
    }
    request.getMapBean().put("accountId", account.getId());

    Page<AccountGroup> page = accountService.selectAccountGroups(request.getPageIndex(),
        request.getPageSize(), request.getMapBean());

    AccountRelatedGroupsResponseData data = AccountRelatedGroupsResponseData.of(request).build(page);

    return new DefaultApiResponse<AccountRelatedGroupsResponseData>(data);
  }

  /**
   * 
   * curl -i -s -X POST -H 'Content-Type:application/json' -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts/1/groups' \
   * -d '{"groupAccounts":[{"groupId":"1"},{"groupId":"2"}]}'
   * 
   * 
   * @param id
   * @param groupAccounts
   * @return
   */
  @RequestMapping(method = RequestMethod.POST, path = "/{id}/groups", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<AccountRelateGroupsResponseData> relateGroups(
      @PathVariable("id") String id, 
      @RequestBody AccountRelateGroupsRequest accountGroups) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Account tmp = accountService.selectById(id);

    if (tmp == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    accountService.relateAccountGroups(tmp, accountGroups.getAccountGroups());

    AccountRelateGroupsResponseData data = AccountRelateGroupsResponseData.of("info.relate.success");

    return new DefaultApiResponse<AccountRelateGroupsResponseData>(data);
  }

  /**
   * 
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts/1/roles'
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts/1/roles?pageIndex=2&pageSize=50'
   * curl -i -s -X GET -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts/1/roles?pageIndex=0&pageSize=20&mapBean[roleCode]=roleCode&mapBean[roleName]=roleName'
   * 
   * 
   * 
   * @param id
   * @param pagerRequestModel
   * @return
   */
  @RequestMapping(method = RequestMethod.GET, path = "/{id}/roles", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<AccountRelatedRolesResponseData> accountRoles(
      @PathVariable("id") String id, 
      AccountRelatedRolesRequest request) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Account account = accountService.selectById(id);

    if (account == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    if (request.getMapBean() == null) {
      request.setMapBean(new HashMap<String, Object>());
    }
    request.getMapBean().put("accountId", account.getId());

    Page<AccountRole> page = accountService.selectAccountRoles(request.getPageIndex(),
        request.getPageSize(), request.getMapBean());

    AccountRelatedRolesResponseData data = AccountRelatedRolesResponseData.of(request).build(page);

    return new DefaultApiResponse<AccountRelatedRolesResponseData>(data);
  }

  /**
   * 
   * curl -i -s -X POST -H 'Content-Type:application/json' -H 'Accept:application/json' 'http://localhost:8081/api/v1/admin/accounts/1/roles' \
   * -d '{"accountRoles":[{"roleId":"1"},{"roleId":"2"}]}'
   * 
   * 
   * @param id
   * @param accountRoles
   * @return
   */
  @RequestMapping(method = RequestMethod.POST, path = "/{id}/roles", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<AccountRelateRolesResponseData> relateRoles(
      @PathVariable("id") String id, 
      @RequestBody AccountRelateRolesRequest accountRoles) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Account account = accountService.selectById(id);

    if (account == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    accountService.relateAccountRoles(account, accountRoles.getAccountRoles());

    AccountRelateRolesResponseData data = AccountRelateRolesResponseData.of("info.relate.success");

    return new DefaultApiResponse<AccountRelateRolesResponseData>(data);
  }

}
