package com.supwisdom.institute.backend.system.api.v1.admin;

import java.util.HashMap;
import java.util.List;

import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.vo.response.DefaultApiResponse;
import com.supwisdom.institute.backend.system.api.vo.request.RoleCreateRequest;
import com.supwisdom.institute.backend.system.api.vo.request.RoleDeleteBatchRequest;
import com.supwisdom.institute.backend.system.api.vo.request.RoleRelateAccountsRequest;
import com.supwisdom.institute.backend.system.api.vo.request.RoleRelateGroupsRequest;
import com.supwisdom.institute.backend.system.api.vo.request.RoleRelatePermissionsRequest;
import com.supwisdom.institute.backend.system.api.vo.request.RoleRelatedAccountsRequest;
import com.supwisdom.institute.backend.system.api.vo.request.RoleRelatedGroupsRequest;
import com.supwisdom.institute.backend.system.api.vo.request.RoleRelatedPermissionsRequest;
import com.supwisdom.institute.backend.system.api.vo.request.RoleUpdateRequest;
import com.supwisdom.institute.backend.system.api.vo.request.RoleQueryRequest;
import com.supwisdom.institute.backend.system.api.vo.response.RoleCreateResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleDeleteBatchResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleLoadResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleRelateAccountsResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleRelateGroupsResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleRelatePermissionsResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleRelatedAccountsResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleRelatedGroupsResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleRelatedPermissionsResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleRemoveResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleUpdateResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.RoleQueryResponseData;
import com.supwisdom.institute.backend.system.domain.entity.AccountRole;
import com.supwisdom.institute.backend.system.domain.entity.GroupRole;
import com.supwisdom.institute.backend.system.domain.entity.Role;
import com.supwisdom.institute.backend.system.domain.entity.RolePermission;
import com.supwisdom.institute.backend.system.domain.service.RoleService;

@Api(value = "SystemAdminRole", tags = { "SystemAdminRole" }, description = "角色的操作接口")
@Slf4j
@RestController
@RequestMapping("/v1/admin/roles")
public class AdminRoleController {

  @Autowired
  private RoleService roleService;

  @GetMapping(produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RoleQueryResponseData> query(RoleQueryRequest queryRequest) {

    Page<Role> page = roleService.selectPageList(
        queryRequest.isLoadAll(), 
        queryRequest.getPageIndex(), 
        queryRequest.getPageSize(),
        queryRequest.getMapBean(),
        queryRequest.getOrderBy());

    RoleQueryResponseData data = RoleQueryResponseData.of(queryRequest).build(page);

    return new DefaultApiResponse<RoleQueryResponseData>(data);
  }
  
  @GetMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RoleLoadResponseData> load(@PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Role role = roleService.selectById(id);

    if (role == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }
    
    RoleLoadResponseData data = RoleLoadResponseData.of(role);

    return new DefaultApiResponse<RoleLoadResponseData>(data);
  }

  @PostMapping(consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RoleCreateResponseData> create(
      @RequestBody RoleCreateRequest createRequest) {
    
    // FIXME: 验证数据有效性
    
    Role role = createRequest.getEntity();
    
    Role ret = roleService.insert(role);
    
    RoleCreateResponseData data = RoleCreateResponseData.build(ret);

    return new DefaultApiResponse<RoleCreateResponseData>(data);
  }
  
  @PutMapping(path = "/{id}", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RoleUpdateResponseData> update(
      @PathVariable("id") String id, 
      @RequestBody RoleUpdateRequest updateRequest) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.update.id.must.not.empty");
    }

    Role tmp = roleService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.update.domain.not.exist");
    }

    Role role = updateRequest.getEntity();
    role.setId(id);

    role = EntityUtils.merge(tmp, role);

    Role ret = roleService.update(role);

    RoleUpdateResponseData data = RoleUpdateResponseData.build(ret);
    
    return new DefaultApiResponse<RoleUpdateResponseData>(data);
  }

  @DeleteMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RoleRemoveResponseData> delete(
      @PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.delete.id.must.not.empty"); // FIXME: RestException
    }

    Role tmp = roleService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.delete.domain.not.exist"); // FIXME: RestException
    }

    roleService.deleteById(id);

    RoleRemoveResponseData data = RoleRemoveResponseData.build(tmp);
    return new DefaultApiResponse<RoleRemoveResponseData>(data);
  }

  @DeleteMapping(path = "/batch", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<RoleDeleteBatchResponseData> deleteBatch(
      @RequestBody RoleDeleteBatchRequest deleteBatchRequest) {
    
    System.out.println(deleteBatchRequest.getIds());
    List<String> ids = deleteBatchRequest.getIds();
    
    roleService.deleteBatch(ids);
    
    RoleDeleteBatchResponseData data = RoleDeleteBatchResponseData.build(ids);
    return new DefaultApiResponse<RoleDeleteBatchResponseData>(data);
  }


  @RequestMapping(method = RequestMethod.GET, path = "/{id}/accounts", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<RoleRelatedAccountsResponseData> roleAccounts(
      @PathVariable("id") String id, 
      RoleRelatedAccountsRequest request) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Role role = roleService.selectById(id);

    if (role == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    if (request.getMapBean() == null) {
      request.setMapBean(new HashMap<String, Object>());
    }
    request.getMapBean().put("roleId", role.getId());

    Page<AccountRole> page = roleService.selectRoleAccounts(
        request.getPageIndex(),
        request.getPageSize(), 
        request.getMapBean());

    RoleRelatedAccountsResponseData data = RoleRelatedAccountsResponseData.of(request).build(page);

    return new DefaultApiResponse<RoleRelatedAccountsResponseData>(data);
  }

  @RequestMapping(method = RequestMethod.POST, path = "/{id}/accounts", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<RoleRelateAccountsResponseData> relateAccounts(
      @PathVariable("id") String id, 
      @RequestBody RoleRelateAccountsRequest roleAccounts) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Role role = roleService.selectById(id);

    if (role == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    roleService.relateRoleAccounts(role, roleAccounts.getRoleAccounts());

    RoleRelateAccountsResponseData data = RoleRelateAccountsResponseData.of("info.relate.success");

    return new DefaultApiResponse<RoleRelateAccountsResponseData>(data);
  }


  @RequestMapping(method = RequestMethod.GET, path = "/{id}/groups", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<RoleRelatedGroupsResponseData> roleGroups(
      @PathVariable("id") String id, 
      RoleRelatedGroupsRequest request) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Role role = roleService.selectById(id);

    if (role == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    if (request.getMapBean() == null) {
      request.setMapBean(new HashMap<String, Object>());
    }
    request.getMapBean().put("roleId", role.getId());

    Page<GroupRole> page = roleService.selectRoleGroups(
        request.getPageIndex(),
        request.getPageSize(), 
        request.getMapBean());

    RoleRelatedGroupsResponseData data = RoleRelatedGroupsResponseData.of(request).build(page);

    return new DefaultApiResponse<RoleRelatedGroupsResponseData>(data);
  }

  @RequestMapping(method = RequestMethod.POST, path = "/{id}/groups", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<RoleRelateGroupsResponseData> relateGroups(
      @PathVariable("id") String id, 
      @RequestBody RoleRelateGroupsRequest roleGroups) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Role role = roleService.selectById(id);

    if (role == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    roleService.relateRoleGroups(role, roleGroups.getRoleGroups());

    RoleRelateGroupsResponseData data = RoleRelateGroupsResponseData.of("info.relate.success");

    return new DefaultApiResponse<RoleRelateGroupsResponseData>(data);
  }


  @RequestMapping(method = RequestMethod.GET, path = "/{id}/permissions", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<RoleRelatedPermissionsResponseData> rolePermissions(
      @PathVariable("id") String id, 
      RoleRelatedPermissionsRequest request) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Role role = roleService.selectById(id);

    if (role == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    if (request.getMapBean() == null) {
      request.setMapBean(new HashMap<String, Object>());
    }
    request.getMapBean().put("roleId", role.getId());

    Page<RolePermission> page = roleService.selectRolePermissions(
        request.getPageIndex(),
        request.getPageSize(), 
        request.getMapBean());

    RoleRelatedPermissionsResponseData data = RoleRelatedPermissionsResponseData.of(request).build(page);

    return new DefaultApiResponse<RoleRelatedPermissionsResponseData>(data);
  }

  @RequestMapping(method = RequestMethod.POST, path = "/{id}/permissions", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseBody
  public DefaultApiResponse<RoleRelatePermissionsResponseData> relatePermissions(
      @PathVariable("id") String id, 
      @RequestBody RoleRelatePermissionsRequest rolePermissions) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Role role = roleService.selectById(id);

    if (role == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }

    roleService.relateRolePermissions(role, rolePermissions.getRolePermissions());

    RoleRelatePermissionsResponseData data = RoleRelatePermissionsResponseData.of("info.relate.success");

    return new DefaultApiResponse<RoleRelatePermissionsResponseData>(data);
  }

}
