package com.supwisdom.institute.backend.system.domain.repo;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.transaction.Transactional;

import org.springframework.data.domain.Example;
import org.springframework.data.domain.ExampleMatcher;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;
import org.springframework.util.StringUtils;

import com.supwisdom.institute.backend.common.framework.repo.BaseJpaRepository;
import com.supwisdom.institute.backend.common.util.MapBeanUtils;
import com.supwisdom.institute.backend.system.domain.entity.Permission;

@Repository
@Transactional
public interface PermissionRepository extends BaseJpaRepository<Permission> {
  

  @Override
  public default Specification<Permission> convertToSpec(Map<String, Object> mapBean) {
    
    Specification<Permission> spec = new Specification<Permission>() {

      /**
       * 
       */
      private static final long serialVersionUID = 9071470982419099273L;

      @Override
      public Predicate toPredicate(Root<Permission> root, CriteriaQuery<?> query, CriteriaBuilder criteriaBuilder) {
        List<Predicate> predicates = new ArrayList<>();
        
        if (mapBean != null) {

          if (MapBeanUtils.getBoolean(mapBean, "deleted") != null) {
            predicates.add(criteriaBuilder.equal(root.get("deleted"), MapBeanUtils.getBoolean(mapBean, "deleted")));
          }
          
          if (!StringUtils.isEmpty(MapBeanUtils.getString(mapBean, "code"))) {
            predicates.add(criteriaBuilder.like(root.get("code"), "%" + MapBeanUtils.getString(mapBean, "code") + "%"));
          }
          if (!StringUtils.isEmpty(MapBeanUtils.getString(mapBean, "name"))) {
            predicates.add(criteriaBuilder.like(root.get("name"), "%" + MapBeanUtils.getString(mapBean, "name") + "%"));
          }
          
          if (!StringUtils.isEmpty(MapBeanUtils.getString(mapBean, "status"))) {
            predicates.add(criteriaBuilder.equal(root.get("status"), MapBeanUtils.getString(mapBean, "status")));
          }
          
          if (!StringUtils.isEmpty(MapBeanUtils.getString(mapBean, "type"))) {
            predicates.add(criteriaBuilder.equal(root.get("type"), MapBeanUtils.getString(mapBean, "type")));
          }
          
          if (!StringUtils.isEmpty(MapBeanUtils.getString(mapBean, "url"))) {
            predicates.add(criteriaBuilder.like(root.get("url"), "%" + MapBeanUtils.getString(mapBean, "url") + "%"));
          }

          if (!StringUtils.isEmpty(MapBeanUtils.getString(mapBean, "parentId"))) {
            predicates.add(criteriaBuilder.equal(root.get("parentId"), MapBeanUtils.getString(mapBean, "parentId")));
          }

//          if (!StringUtils.isEmpty(MapBeanUtils.getString(mapBean, "grantTimeBegin"))) {
//            String grantTimeBegin = MapBeanUtils.getString(mapBean, "grantTimeBegin");
//            Date d = DateUtil.parseDate(grantTimeBegin+" 00:00:00", "yyyy-MM-dd HH:mm:ss");
//            
//            if (d != null) {
//              predicates.add(criteriaBuilder.greaterThanOrEqualTo(root.get("grantTime"), d));
//            }
//          }
//
//          if (!StringUtils.isEmpty(MapBeanUtils.getString(mapBean, "grantTimeEnd"))) {
//            String grantTimeEnd = MapBeanUtils.getString(mapBean, "grantTimeEnd");
//            Date d = DateUtil.parseDate(grantTimeEnd+" 23:59:59", "yyyy-MM-dd HH:mm:ss");
//            
//            if (d != null) {
//              predicates.add(criteriaBuilder.lessThanOrEqualTo(root.get("grantTime"), d));
//            }
//          }

          List<Predicate> predicatesKeyword = new ArrayList<>();
          if (!StringUtils.isEmpty(MapBeanUtils.getString(mapBean, "keyword"))) {
            predicatesKeyword.add(criteriaBuilder.like(root.get("username"), "%" + MapBeanUtils.getString(mapBean, "keyword") + "%"));
            predicatesKeyword.add(criteriaBuilder.like(root.get("name"), "%" + MapBeanUtils.getString(mapBean, "keyword") + "%"));
            predicatesKeyword.add(criteriaBuilder.like(root.get("memo"), "%" + MapBeanUtils.getString(mapBean, "keyword") + "%"));
            
            predicates.add(criteriaBuilder.or(predicatesKeyword.toArray(new Predicate[predicatesKeyword.size()])));
          }
        }
        
        return criteriaBuilder.and(predicates.toArray(new Predicate[predicates.size()]));
      }
      
    };
    
    return spec;
  }
  
  @Override
  public default Page<Permission> selectPageList(boolean loadAll, int pageIndex, int pageSize, Map<String, Object> mapBean, Map<String, String> orderBy) {
    if (loadAll) {
      pageIndex = 0;
      pageSize = Integer.MAX_VALUE;
    }
    
    Permission probe = new Permission();
    if (mapBean != null) {
      probe.setDeleted(MapBeanUtils.getBoolean(mapBean, "deleted"));
      probe.setCode(MapBeanUtils.getString(mapBean, "code"));
      probe.setName(MapBeanUtils.getString(mapBean, "name"));
      probe.setMemo(MapBeanUtils.getString(mapBean, "memo"));
      probe.setStatus(MapBeanUtils.getString(mapBean, "status"));
      probe.setType(MapBeanUtils.getString(mapBean, "type"));
    }
    
    ExampleMatcher matcher = ExampleMatcher.matching()
        .withMatcher("deleted", ExampleMatcher.GenericPropertyMatchers.exact())
        .withMatcher("code", ExampleMatcher.GenericPropertyMatchers.contains())
        .withMatcher("name", ExampleMatcher.GenericPropertyMatchers.contains())
        .withMatcher("memo", ExampleMatcher.GenericPropertyMatchers.contains())
        .withMatcher("status", ExampleMatcher.GenericPropertyMatchers.exact())
        .withMatcher("type", ExampleMatcher.GenericPropertyMatchers.exact());
    
    PageRequest pageRequest = PageRequest.of(pageIndex, pageSize);
    Example<Permission> example = Example.of(probe, matcher);
    
    Page<Permission> page = this.findAll(example, pageRequest);
    
    return page;
  }
  
  
  
  @Query(value = "select max(p.rgt) from Permission p")
  public int selectMaxRgt();
  
  @Query(value = "select p from Permission p where p.lft>:lft and p.rgt<:rgt order by p.lft")
  public List<Permission> selectBetweenLftRgt(@Param("lft") int lft, @Param("rgt") int rgt);
  
  @Modifying
  @Query(value = "update TB_U_PERMISSION "
      + "set "
      + "  LFT = (case when LFT >= :rgt then LFT + :offset else LFT end), "
      + "  RGT = RGT + :offset "
      + "where RGT >= :rgt", nativeQuery = true)
  public int updateLftRgtWhenInsert(@Param("rgt") int rgt, @Param("offset") int offset);

  @Modifying
  @Query(value = "update TB_U_PERMISSION "
      + "set "
      + "  LFT = (case when LFT >= :rgt then LFT - :offset else LFT end), "
      + "  RGT = RGT - :offset "
      + "where RGT >= :rgt", nativeQuery = true)
  public int updateLftRgtWhenDelete(@Param("rgt") int rgt, @Param("offset") int offset);
  
  
  @Override
  public default Permission insert(Permission entity) {

    if (entity.getParentId() == null) {
      entity.setParentId("0");
    }

    if (entity.getParentId() == null || entity.getParentId().isEmpty() || "0".equals(entity.getParentId())) {
      int maxRgt = selectMaxRgt();
      entity.setLft((maxRgt+1));
      entity.setRgt((maxRgt+1) + 1);
      
      entity.setLevel(1);
    } else {
      Permission parentEntity = this.selectById(entity.getParentId());
      if (parentEntity == null) {
        throw new RuntimeException(String.format("父级对象不存在！"));
      } else {
        // 将 lft或rgt 大于等于父级对象 rgt 的记录的 lft、rgt +offset
        int rgt = parentEntity.getRgt();
        int offset = 2;
        updateLftRgtWhenInsert(rgt, offset);
        
        entity.setLft(rgt);
        entity.setRgt(rgt + 1);
        
        entity.setLevel(parentEntity.getLevel() + 1);
      }
    }
    
    return BaseJpaRepository.super.insert(entity);
  }
  
  @Override
  public default Permission update(Permission entity) {

    Permission originEntity = this.selectById(entity.getId());
    if (originEntity == null) {
      return null;
    }
    
    //if (!this.checkFieldExists("code", entity.getCode(), entity.getId())) {
    //  throw new RuntimeException(String.format("代码重复！"));
    //}
    
    if (originEntity.getParentId() == null) {
      originEntity.setParentId("0");
    }
    
    if (entity.getParentId() == null) {
      entity.setParentId("0");
    }
    
    if (!originEntity.getParentId().equals(entity.getParentId()) ) {

      int lft = originEntity.getLft();
      int rgt = originEntity.getRgt();
      int level = originEntity.getLevel();
      int offset = rgt - lft +1;
      
      List<Permission> childEntities = this.selectBetweenLftRgt(lft, rgt);

      if (entity.getParentId() == null || entity.getParentId().isEmpty() || "0".equals(entity.getParentId())) {
        // 将 lft或rgt 大于等于该对象 rgt 的记录的 lft、rgt -offset
        updateLftRgtWhenDelete(rgt, offset);
  
        int maxRgt = selectMaxRgt();
        entity.setLft((maxRgt+1));
        entity.setRgt((maxRgt+1) + 1 +offset-2);
        
        entity.setLevel(1);
      } else {
        // 将 lft或rgt 大于等于该对象 rgt 的记录的 lft、rgt -offset
        updateLftRgtWhenDelete(rgt, offset);

        Permission parentEntity = this.selectById(entity.getParentId());
        if (parentEntity == null) {
          throw new RuntimeException(String.format("父级对象不存在！"));
        }
        //System.out.println(String.format("pLft %s, pRgt %s", parentEntity.getLft(), parentEntity.getRgt()));
        if (parentEntity.getLft() >= originEntity.getLft() && parentEntity.getRgt() <= originEntity.getRgt()) {
          throw new RuntimeException(String.format("不能设置自身或自身的子节点作为父级！"));
        }
        
        //parentEntity = this.selectById(entity.getParentId()); System.out.println(String.format("pLft %s, pRgt %s", parentEntity.getLft(), parentEntity.getRgt()));
        // 将 lft或rgt 大于等于父级对象 rgt 的记录的 lft、rgt +offset
        //int pLft = parentEntity.getLft();
        int pRgt = parentEntity.getRgt();
        updateLftRgtWhenInsert(pRgt, offset);
        
        entity.setLft(pRgt);
        entity.setRgt(pRgt + 1 + offset-2);
        
        entity.setLevel(parentEntity.getLevel() + 1);
      }
      
      int newLft = entity.getLft();
      int newRgt = entity.getRgt();
      int newLevel = entity.getLevel();
      //System.out.println(String.format("newLft %s, newRgt %s, newLevel %s", newLft, newRgt, newLevel));
      //System.out.println(String.format("lft %s, rgt %s, level %s", lft, rgt, level));
      for (Permission childEntity : childEntities) {
        //Permission pEntity = this.selectById(childEntity.getParentId());
        
        int cLft = childEntity.getLft();
        int cRgt = childEntity.getRgt();
        int cLevel = childEntity.getLevel();
        
        childEntity.setLft(cLft + (newLft - lft));
        childEntity.setRgt(cRgt + (newRgt - rgt));
        
        childEntity.setLevel(cLevel + (newLevel - level));
        
        BaseJpaRepository.super.update(childEntity);
      }

    }
    
    return BaseJpaRepository.super.update(entity);
  }

  @Override
  public default void delete(String id) {
    
    Permission originEntity = this.selectById(id);
    if (originEntity == null) {
      return;
    }

    int lft = originEntity.getLft();
    int rgt = originEntity.getRgt();
    int offset = rgt - lft +1;

    // FIXME: 判断是否有子节点
    //if (lft + 1 != rgt) {
    //  return;
    //}

    List<Permission> childEntities = this.selectBetweenLftRgt(lft, rgt);
    for (Permission childEntity : childEntities) {
      BaseJpaRepository.super.delete(childEntity.getId());
    }
    
    // 将 lft或rgt 大于等于该对象 rgt 的记录的 lft、rgt -offset
    updateLftRgtWhenDelete(rgt, offset);
    
    BaseJpaRepository.super.delete(id);
  }

  public default Permission selectApplicationPermissionByCode(String code) {
    Permission probe = new Permission();
    probe.setCode(code);
    probe.setType("1");
    
    ExampleMatcher matcher = ExampleMatcher.matching()
        .withMatcher("code", ExampleMatcher.GenericPropertyMatchers.exact())
        .withMatcher("type", ExampleMatcher.GenericPropertyMatchers.exact());
    
    Example<Permission> example = Example.of(probe, matcher);
    
    Optional<Permission> o = this.findOne(example);
    
    if (o.isPresent()) {
      return o.get();
    }
    
    return null;
  }
  
 

  @Query(value = "select p from Permission p "
      + "inner join RolePermission rp on p.id=rp.permissionId "
      + "inner join Role r on rp.roleId=r.id "
      + "inner join AccountRole ar on r.id=ar.roleId "
      + "inner join Account a on ar.accountId=a.id "
      + "where a.username=:username "
      + "and p.lft >= :lft and p.rgt <= :rgt "
      + "and (:type is null or p.type=:type) "
      + "and p.status='1' and r.status='1' and a.status='1' and a.enabled=1 ")
  public List<Permission> selectAccountRolePermissionByUsername(@Param("username") String username, @Param("lft") int lft, @Param("rgt") int rgt, @Param("type") String type);
  
  @Query(value = "select p from Permission p "
      + "inner join RolePermission rp on p.id=rp.permissionId "
      + "inner join Role r on rp.roleId=r.id "
      + "inner join GroupRole gr on r.id=gr.roleId "
      + "inner join Group_ g on gr.groupId=g.id "
      + "inner join AccountGroup ag on g.id=ag.groupId "
      + "inner join Account a on ag.accountId=a.id "
      + "where a.username=:username "
      + "and p.lft >= :lft and p.rgt <= :rgt "
      + "and (:type is null or p.type=:type) "
      + "and p.status='1' and r.status='1' and g.status='1' and a.status='1' and a.enabled=1 ")
  public List<Permission> selectAccountGroupRolePermissionByUsername(@Param("username") String username, @Param("lft") int lft, @Param("rgt") int rgt, @Param("type") String type);

  public default List<Permission> selectByUsername(String username, String applicationCode, String type) {
    List<Permission> permissions = new ArrayList<Permission>();
    
    Permission applicationPermission = selectApplicationPermissionByCode(applicationCode);
    if (applicationPermission == null) {
      return permissions;
    }
    
    int lft = applicationPermission.getLft();
    int rgt = applicationPermission.getRgt();
    
    List<Permission> accountRolePermissions = selectAccountRolePermissionByUsername(username, lft, rgt, type);
    permissions.addAll(accountRolePermissions);
    
    List<Permission> accountGroupRolePermissions = selectAccountGroupRolePermissionByUsername(username, lft, rgt, type);
    permissions.addAll(accountGroupRolePermissions);
    
    return permissions;
  }




  public default List<Permission> selectList(boolean loadAll, int pageIndex, int pageSize, Map<String, Object> mapBean, Map<String, String> orderBy) {
    
    Specification<Permission> spec = convertToSpec(mapBean);
    
    if (loadAll) {
      pageIndex = 0;
      pageSize = Integer.MAX_VALUE;
    }
    
    Sort sort = convertToSort(orderBy);

    if (sort == null) {
      return this.findAll(spec);
    } else {
      return this.findAll(spec, sort);
    }
  }


}
