package com.supwisdom.institute.backend.system.api.v1.admin;

import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.HttpStatus;
import org.springframework.util.MimeTypeUtils;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.vo.response.DefaultApiResponse;
import com.supwisdom.institute.backend.system.api.vo.request.ResourceCreateRequest;
import com.supwisdom.institute.backend.system.api.vo.request.ResourceQueryRequest;
import com.supwisdom.institute.backend.system.api.vo.request.ResourceUpdateRequest;
import com.supwisdom.institute.backend.system.api.vo.response.ResourceCreateResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.ResourceLoadResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.ResourceQueryResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.ResourceRemoveResponseData;
import com.supwisdom.institute.backend.system.api.vo.response.ResourceUpdateResponseData;
import com.supwisdom.institute.backend.system.domain.entity.Resource;
import com.supwisdom.institute.backend.system.domain.service.ResourceService;

@Api(value = "SystemAdminResource", tags = { "SystemAdminResource" }, description = "资源（API）的操作接口")
@Slf4j
@RestController
@RequestMapping("/v1/admin/resources")
public class AdminResourceController {

  @Autowired
  private ResourceService resourceService;

  @GetMapping(produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ResourceQueryResponseData> query(ResourceQueryRequest queryRequest) {

    Page<Resource> page = resourceService.selectPageList(
        queryRequest.isLoadAll(), 
        queryRequest.getPageIndex(), 
        queryRequest.getPageSize(),
        queryRequest.getMapBean(),
        queryRequest.getOrderBy());

    ResourceQueryResponseData data = ResourceQueryResponseData.of(queryRequest).build(page);

    return new DefaultApiResponse<ResourceQueryResponseData>(data);
  }
  
  @GetMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ResourceLoadResponseData> load(@PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.get.id.must.not.empty"); // FIXME: RestException
    }

    Resource resource = resourceService.selectById(id);

    if (resource == null) {
      throw new RuntimeException("exception.get.domain.not.exist"); // FIXME: RestException
    }
    
    ResourceLoadResponseData data = ResourceLoadResponseData.of(resource);

    return new DefaultApiResponse<ResourceLoadResponseData>(data);
  }

  @PostMapping(consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ResourceCreateResponseData> create(
      @RequestBody ResourceCreateRequest createRequest) {
    
    // FIXME: 验证数据有效性
    
    Resource resource = createRequest.getEntity();
    
    Resource ret = resourceService.insert(resource);
    
    ResourceCreateResponseData data = ResourceCreateResponseData.build(ret);

    return new DefaultApiResponse<ResourceCreateResponseData>(data);
  }
  
  @PutMapping(path = "/{id}", consumes = MimeTypeUtils.APPLICATION_JSON_VALUE, produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ResourceUpdateResponseData> update(
      @PathVariable("id") String id, 
      @RequestBody ResourceUpdateRequest updateRequest) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.update.id.must.not.empty");
    }

    Resource tmp = resourceService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.update.domain.not.exist");
    }

    Resource resource = updateRequest.getEntity();
    resource.setId(id);

    resource = EntityUtils.merge(tmp, resource);

    Resource ret = resourceService.update(resource);

    ResourceUpdateResponseData data = ResourceUpdateResponseData.build(ret);
    
    return new DefaultApiResponse<ResourceUpdateResponseData>(data);
  }

  @DeleteMapping(path = "/{id}", produces = MimeTypeUtils.APPLICATION_JSON_VALUE)
  @ResponseStatus(value = HttpStatus.OK)
  @ResponseBody
  public DefaultApiResponse<ResourceRemoveResponseData> delete(
      @PathVariable("id") String id) {

    if (id == null || id.length() == 0) {
      throw new RuntimeException("exception.delete.id.must.not.empty"); // FIXME: RestException
    }

    Resource tmp = resourceService.selectById(id);
    if (tmp == null) {
      throw new RuntimeException("exception.delete.domain.not.exist"); // FIXME: RestException
    }

    resourceService.deleteById(id);

    ResourceRemoveResponseData data = ResourceRemoveResponseData.build(tmp);
    return new DefaultApiResponse<ResourceRemoveResponseData>(data);
  }

}
