package com.supwisdom.institute.backend.common.framework.repo;

import java.util.Calendar;
import java.util.Map;
import java.util.Optional;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.repository.NoRepositoryBean;

import com.supwisdom.institute.backend.common.core.transmit.user.UserContext;
import com.supwisdom.institute.backend.common.framework.entity.ABaseEntity;
import com.supwisdom.institute.backend.common.util.UUIDUtils;

@NoRepositoryBean
public interface BaseJpaRepository<E extends ABaseEntity> extends JpaRepository<E, String>, JpaSpecificationExecutor<E> {
  

  /**
   * 生成主键值。 默认使用方法
   * 如果需要生成主键，需要由子类重写此方法根据需要的方式生成主键值。 
   * @param entity 要持久化的对象 
   */
  public default String generateId() {
    return UUIDUtils.create();
  }
  
  public default Specification<E> convertToSpec(Map<String, Object> mapBean) {
    return null;
  }
  public default Sort convertToSort(Map<String, String> orderBy) {
    return null;
  }

  public default Page<E> selectPageList(boolean loadAll, int pageIndex, int pageSize, Map<String, Object> mapBean, Map<String, String> orderBy) {
    
    Specification<E> spec = convertToSpec(mapBean);
    
    if (pageIndex == -1) {
      loadAll = true;
    }
    
    if (loadAll) {
      pageIndex = 0;
      pageSize = Integer.MAX_VALUE;
    }
    
    Sort sort = convertToSort(orderBy);

    PageRequest pageRequest = PageRequest.of(pageIndex, pageSize);
    if (sort != null) {
      pageRequest = PageRequest.of(pageIndex, pageSize, sort);
    }

    Page<E> page = this.findAll(spec, pageRequest);

    return page;
  }
  



  public default E selectById(String id) {

    try {
      Optional<E> entity = this.findById(id);

      if (entity.isPresent()) {
        return entity.get();
      }
    } catch (RuntimeException e) {
      System.out.println("RuntimeException:" + e.getMessage());
    } catch (Exception e) {
      System.out.println("Exception:" + e.getMessage());
    }

    return null;
  }
  
  public default E insert(E entity) {
    
    if (entity.getId() == null || entity.getId().isEmpty()) {
      entity.setId(generateId());
    }

    if (entity.getCompanyId() == null || entity.getCompanyId().isEmpty()) {
      entity.setCompanyId("1");
    }

    if (entity.getDeleted() == null) {
      entity.setDeleted(false);
    }
    if (entity.getAddAccount() == null) {
      entity.setAddAccount(UserContext.getUsername());
    }
    if (entity.getAddTime() == null) {
      entity.setAddTime(Calendar.getInstance().getTime());
    }
    
    E e = this.save(entity);

    return e;
  }
  
  public default E update(E entity) {

    if (entity.getEditAccount() == null) {
      entity.setEditAccount(UserContext.getUsername());
    }
    if (entity.getEditTime() == null) {
      entity.setEditTime(Calendar.getInstance().getTime());
    }
    
    E e = this.save(entity);

    return e;
  }
  
  public default E remove(E entity) {
    
    if (entity.getDeleted() == null) {
      entity.setDeleted(true);
    }
    if (entity.getDeleteAccount() == null) {
      entity.setDeleteAccount(UserContext.getUsername());
    }
    if (entity.getDeleteTime() == null) {
      entity.setDeleteTime(Calendar.getInstance().getTime());
    }
    
    E e = this.save(entity);
    
    return e;
  }
  
  public default void delete(String id) {
    
    this.deleteById(id);
  }

}
