package com.supwisdom.institute.backend.base.domain.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Example;
import org.springframework.data.domain.ExampleMatcher;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import com.supwisdom.institute.backend.base.domain.entity.Permission;
import com.supwisdom.institute.backend.base.domain.entity.PermissionResource;
import com.supwisdom.institute.backend.base.domain.entity.Role;
import com.supwisdom.institute.backend.base.domain.model.PermissionRoleSet;
import com.supwisdom.institute.backend.base.domain.model.PermissionTreeNode;
import com.supwisdom.institute.backend.base.domain.repo.PermissionRepository;
import com.supwisdom.institute.backend.base.domain.repo.PermissionResourceRepository;
import com.supwisdom.institute.backend.base.domain.repo.RoleRepository;
import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.service.ABaseService;
import com.supwisdom.institute.backend.common.util.MapBeanUtils;

@Slf4j
@Service
public class PermissionService extends ABaseService<Permission, PermissionRepository> {
  
  @Override
  public PermissionRepository getRepo() {
    return permissionRepository;
  }

  @Autowired
  private PermissionRepository permissionRepository;

  @Autowired
  private PermissionResourceRepository permissionResourceRepository;

  @Autowired
  private RoleRepository roleRepository;

  @Override
  public Permission insert(Permission entity) {
    Permission parentPermission = permissionRepository.selectById(entity.getParentId());
    if (parentPermission != null) {
      entity.setApplicationId(parentPermission.getApplicationId());
    }
    
    return super.insert(entity);
  }

  public void deleteBatch(List<String> ids) {
    
    ids.stream().forEach(id -> {
      this.deleteById(id);
    });
  }
  
  

  public Page<PermissionResource> selectPermissionResources(boolean loadAll, int pageIndex, int pageSize, Map<String, Object> mapBean) {
    
    if (loadAll) {
      pageIndex = 0;
      pageSize = Integer.MAX_VALUE;
    }

    PermissionResource probe = new PermissionResource();
    if (mapBean != null) {
      probe.setPermissionId(MapBeanUtils.getString(mapBean, "permissionId"));
      probe.setResourceId(MapBeanUtils.getString(mapBean, "resourceId"));
    }

    ExampleMatcher matcher = ExampleMatcher.matching()
        .withMatcher("permissionId", ExampleMatcher.GenericPropertyMatchers.exact())
        .withMatcher("resourceId", ExampleMatcher.GenericPropertyMatchers.exact());

    Example<PermissionResource> example = Example.of(probe, matcher);

    PageRequest pageRequest = PageRequest.of(pageIndex, pageSize);

    Page<PermissionResource> page = permissionResourceRepository.findAll(example, pageRequest); // FIXME: 多表关联查询

    return page;
  }

  public void relatePermissionResources(Permission permission, List<PermissionResource> permissionResources) {

    List<PermissionResource> existPermissionResources = this.selectPermissionResourcesByPermissionId(permission.getId());

    Map<String, PermissionResource> existMapPermissionResources = new LinkedHashMap<String, PermissionResource>();
    for (PermissionResource permissionResource : existPermissionResources) {
      String k = String.format("%s", permissionResource.getPermissionId());
      existMapPermissionResources.put(k, permissionResource);
    }

    for (PermissionResource permissionResource : permissionResources) {
      String k = String.format("%s", permissionResource.getPermissionId());

      if (existMapPermissionResources.containsKey(k)) {
        existMapPermissionResources.remove(k);
      } else {
        permissionResource.setCompanyId(permission.getCompanyId());
        permissionResource.setPermissionId(permission.getId());

        permissionResourceRepository.insert(permissionResource);
      }
    }

    for (PermissionResource permissionResource : existMapPermissionResources.values()) {
      permissionResourceRepository.deleteById(permissionResource.getId());
    }
  }

  public List<PermissionResource> selectPermissionResourcesByPermissionId(String permissionId) {

    PermissionResource probe = new PermissionResource();
    probe.setPermissionId(permissionId);

    ExampleMatcher matcher = ExampleMatcher.matching().withMatcher("permissionId",
        ExampleMatcher.GenericPropertyMatchers.exact());

    Example<PermissionResource> example = Example.of(probe, matcher);

    List<PermissionResource> permissionResources = permissionResourceRepository.findAll(example);

    return permissionResources;
  }
  
  
  private Permission selectApplicationPermission() {
    Permission permission = permissionRepository.selectById(Permission.APPLICATION_ID);
    
    return permission;
  }
  
  private List<Permission> selectApplicationPermissionList() {
    Map<String, Object> mapBean = new HashMap<String, Object>();
    mapBean.put("type", Permission.TYPE_APPLICATION);
    
    return permissionRepository.selectList(true, -1, -1, mapBean, null);
  }

  public PermissionTreeNode selectPermissionTree(String applicationId, String type) {

    Map<String, Object> mapBean = new HashMap<String, Object>();
    if (StringUtils.isNotBlank(type)) {
      mapBean.put("type", type);
    }
    
    String rootTreeNodeId = "0";
    
    List<Permission> permissions = new ArrayList<Permission>();
    
    if (StringUtils.isBlank(applicationId)) {
      Permission rootPermission = new Permission();
      rootPermission.setId(Permission.ROOT_PARENT_ID);
      rootPermission.setCode("root");
      rootPermission.setName("根");
      
      permissions.add(rootPermission);
      rootTreeNodeId = rootPermission.getId();

      List<Permission> applicationPermissions = this.selectApplicationPermissionList();
      permissions.addAll(applicationPermissions);
    } else {
      Permission applicationPermission = permissionRepository.selectById(applicationId); // this.selectApplicationPermissionByCode(applicationCode);
      if (applicationPermission == null) {
        return null;
      }
      
      permissions.add(applicationPermission);
      rootTreeNodeId = applicationPermission.getId();
    }
    
    List<Permission> menuPermissions = permissionRepository.selectList(true, -1, -1, mapBean, null);
    permissions.addAll(menuPermissions);

    return convertPermissionTree(permissions, rootTreeNodeId);
  }
  
  private PermissionTreeNode convertPermissionTree(List<Permission> permissions, String rootTreeNodeId) {
    if (permissions == null || permissions.size() == 0) {
      return null;
    }

    Map<String, PermissionTreeNode> parentTreeNode = new LinkedHashMap<String, PermissionTreeNode>();

    for (Permission permission : permissions) {
      PermissionTreeNode treeNode = EntityUtils.copy(permission, new PermissionTreeNode());
      treeNode.setChildren(new ArrayList<PermissionTreeNode>());
      
      if (parentTreeNode.containsKey(treeNode.getId())) {
        continue;
      }
      parentTreeNode.put(treeNode.getId(), treeNode);
    }
    
    for (PermissionTreeNode treeNode : parentTreeNode.values()) {
      if (!parentTreeNode.containsKey(treeNode.getParentId())) {
        continue;
      }
      
      parentTreeNode.get(treeNode.getParentId()).getChildren().add(treeNode);
    }
    
    return parentTreeNode.get(rootTreeNodeId);
  }

  

//  public Permission selectApplicationPermissionByCode(String code) {
//    Permission probe = new Permission();
//    probe.setCode(code);
//    probe.setType("1");
//    
//    ExampleMatcher matcher = ExampleMatcher.matching()
//        .withMatcher("code", ExampleMatcher.GenericPropertyMatchers.exact())
//        .withMatcher("type", ExampleMatcher.GenericPropertyMatchers.exact());
//    
//    Example<Permission> example = Example.of(probe, matcher);
//    
//    Optional<Permission> o = permissionRepository.findOne(example);
//    
//    if (o.isPresent()) {
//      return o.get();
//    }
//    
//    return null;
//  }
  

  public List<Permission> selectByUsername(String username, String applicationId, String type) {
    List<Permission> permissions = new ArrayList<Permission>();
    
//    Permission applicationPermission = permissionRepository.selectById(applicationId);
//    if (applicationPermission == null) {
//      return permissions;
//    }
//    
//    int lft = applicationPermission.getLft();
//    int rgt = applicationPermission.getRgt();
    
    if (StringUtils.isBlank(applicationId)) {
      List<Permission> accountRolePermissions = permissionRepository.selectAccountRolePermissionByUsername(username, type);
      permissions.addAll(accountRolePermissions);
      
      List<Permission> accountGroupRolePermissions = permissionRepository.selectAccountGroupRolePermissionByUsername(username, type);
      permissions.addAll(accountGroupRolePermissions);
    } else {
      List<Permission> accountRolePermissions = permissionRepository.selectAccountRolePermissionByUsername(username, applicationId, type);
      permissions.addAll(accountRolePermissions);
      
      List<Permission> accountGroupRolePermissions = permissionRepository.selectAccountGroupRolePermissionByUsername(username, applicationId, type);
      permissions.addAll(accountGroupRolePermissions);
    }
    
    return EntityUtils.distinctList(permissions);
  }
  
  @Deprecated
  public PermissionTreeNode selectPermissionTreeByUsername(String username, String applicationId, String type) {
    String rootTreeNodeId = "0";
    
    List<Permission> permissions = new ArrayList<Permission>();
    
    if (StringUtils.isBlank(applicationId)) {
      Permission rootPermission = new Permission();
      rootPermission.setId(Permission.ROOT_PARENT_ID);
      rootPermission.setCode("root");
      rootPermission.setName("根");
      
      permissions.add(rootPermission);
      rootTreeNodeId = rootPermission.getId();

      List<Permission> applicationPermissions = this.selectApplicationPermissionList();
      permissions.addAll(applicationPermissions);
    } else {
      Permission applicationPermission = permissionRepository.selectById(applicationId); // this.selectApplicationPermissionByCode(applicationCode);
      if (applicationPermission == null) {
        return null;
      }
      
      permissions.add(applicationPermission);
      rootTreeNodeId = applicationPermission.getId();
    }
    
    List<Permission> menuPermissions = this.selectByUsername(username, applicationId, type);
    permissions.addAll(menuPermissions);

    return convertPermissionTree(permissions, rootTreeNodeId);
  }

  public List<PermissionRoleSet> selectPermissionRoleSet(Map<String, Object> mapBean) {
    
    List<PermissionRoleSet> prsList = new ArrayList<>();
    
    List<Permission> resourceList = permissionRepository.selectList(mapBean, null);
    
    for (Permission permission : resourceList) {
      
      List<Role> roleList = roleRepository.selectPermissionRolesByPermission(permission.getId());
      
      PermissionRoleSet prs = new PermissionRoleSet(permission, roleList);
      
      prsList.add(prs);
    }
    
    return prsList;
  }
  
}
