package com.supwisdom.institute.backend.base.domain.service;

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Example;
import org.springframework.data.domain.ExampleMatcher;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import com.supwisdom.institute.backend.base.domain.entity.Account;
import com.supwisdom.institute.backend.base.domain.entity.AccountGroup;
import com.supwisdom.institute.backend.base.domain.entity.AccountRole;
import com.supwisdom.institute.backend.base.domain.repo.AccountGroupRepository;
import com.supwisdom.institute.backend.base.domain.repo.AccountRepository;
import com.supwisdom.institute.backend.base.domain.repo.AccountRoleRepository;
import com.supwisdom.institute.backend.common.framework.service.ABaseService;
import com.supwisdom.institute.backend.common.util.MapBeanUtils;

@Service
public class AccountService extends ABaseService<Account, AccountRepository> {
  
  @Override
  public AccountRepository getRepo() {
    return accountRepository;
  }

  @Autowired
  private AccountRepository accountRepository;

  @Autowired
  private AccountGroupRepository accountGroupRepository;

  @Autowired
  private AccountRoleRepository accountRoleRepository;
  
  
  @Override
  public Account insert(Account entity) {

    entity.setEnabled(Account.STATUS_ENABLED.equals(entity.getStatus()));
    entity.setAccountNonExpired(true);
    entity.setAccountNonLocked(true);
    entity.setCredentialsNonExpired(true);
    
    return super.insert(entity);
  }
  
  @Override
  public Account update(Account entity) {
    
    entity.setEnabled(Account.STATUS_ENABLED.equals(entity.getStatus()));

    return super.update(entity);
  }
  

  public void deleteBatch(List<String> ids) {
    
    ids.stream().forEach(id -> {
      this.deleteById(id);
    });
  }


  public Page<AccountGroup> selectAccountGroups(int pageIndex, int pageSize, Map<String, Object> mapBean) {

    AccountGroup probe = new AccountGroup();
    if (mapBean != null) {
      probe.setGroupId(MapBeanUtils.getString(mapBean, "groupId"));
      probe.setAccountId(MapBeanUtils.getString(mapBean, "accountId"));
    }

    ExampleMatcher matcher = ExampleMatcher.matching()
        .withMatcher("groupId", ExampleMatcher.GenericPropertyMatchers.exact())
        .withMatcher("accountId", ExampleMatcher.GenericPropertyMatchers.exact());

    Example<AccountGroup> example = Example.of(probe, matcher);

    PageRequest pageRequest = PageRequest.of(pageIndex, pageSize);

    Page<AccountGroup> page = accountGroupRepository.findAll(example, pageRequest); // FIXME: 多表关联查询

    return page;
  }

  public void relateAccountGroups(Account account, List<AccountGroup> accountGroups) {

    List<AccountGroup> existAccountGroups = this.selectAccountGroupsByAccountId(account.getId());

    Map<String, AccountGroup> existMapAccountGroups = new LinkedHashMap<String, AccountGroup>();
    for (AccountGroup accountGroup : existAccountGroups) {
      String k = String.format("%s", accountGroup.getGroupId());
      existMapAccountGroups.put(k, accountGroup);
    }

    for (AccountGroup accountGroup : accountGroups) {
      String k = String.format("%s", accountGroup.getGroupId());

      if (existMapAccountGroups.containsKey(k)) {
        existMapAccountGroups.remove(k);
      } else {
        accountGroup.setCompanyId(account.getCompanyId());
        accountGroup.setAccountId(account.getId());

        accountGroupRepository.insert(accountGroup);
      }
    }

    for (AccountGroup accountGroup : existMapAccountGroups.values()) {
      accountGroupRepository.deleteById(accountGroup.getId());
    }
  }

  public List<AccountGroup> selectAccountGroupsByAccountId(String accountId) {

    AccountGroup probe = new AccountGroup();
    probe.setAccountId(accountId);

    ExampleMatcher matcher = ExampleMatcher.matching()
        .withMatcher("accountId", ExampleMatcher.GenericPropertyMatchers.exact());

    Example<AccountGroup> example = Example.of(probe, matcher);

    List<AccountGroup> accountGroups = accountGroupRepository.findAll(example);

    return accountGroups;
  }

  

  public Page<AccountRole> selectAccountRoles(int pageIndex, int pageSize, Map<String, Object> mapBean) {

    AccountRole probe = new AccountRole();
    if (mapBean != null) {
      probe.setAccountId(MapBeanUtils.getString(mapBean, "accountId"));
      probe.setRoleId(MapBeanUtils.getString(mapBean, "roleId"));
    }

    ExampleMatcher matcher = ExampleMatcher.matching()
        .withMatcher("accountId", ExampleMatcher.GenericPropertyMatchers.exact())
        .withMatcher("roleId", ExampleMatcher.GenericPropertyMatchers.exact());

    Example<AccountRole> example = Example.of(probe, matcher);

    PageRequest pageRequest = PageRequest.of(pageIndex, pageSize);

    Page<AccountRole> page = accountRoleRepository.findAll(example, pageRequest); // FIXME: 多表关联查询

    return page;
  }

  public void relateAccountRoles(Account account, List<AccountRole> accountRoles) {

    List<AccountRole> existAccountRoles = this.selectAccountRolesByAccountId(account.getId());

    Map<String, AccountRole> existMapAccountRoles = new LinkedHashMap<String, AccountRole>();
    for (AccountRole accountRole : existAccountRoles) {
      String k = String.format("%s", accountRole.getRoleId());
      existMapAccountRoles.put(k, accountRole);
    }

    for (AccountRole accountRole : accountRoles) {
      String k = String.format("%s", accountRole.getRoleId());

      if (existMapAccountRoles.containsKey(k)) {
        existMapAccountRoles.remove(k);
      } else {
        accountRole.setCompanyId(account.getCompanyId());
        accountRole.setAccountId(account.getId());

        accountRoleRepository.insert(accountRole);
      }
    }

    for (AccountRole accountRole : existMapAccountRoles.values()) {
      accountRoleRepository.deleteById(accountRole.getId());
    }
  }

  public List<AccountRole> selectAccountRolesByAccountId(String accountId) {

    AccountRole probe = new AccountRole();
    probe.setAccountId(accountId);

    ExampleMatcher matcher = ExampleMatcher.matching()
        .withMatcher("accountId", ExampleMatcher.GenericPropertyMatchers.exact());

    Example<AccountRole> example = Example.of(probe, matcher);

    List<AccountRole> accountRoles = accountRoleRepository.findAll(example);

    return accountRoles;
  }

  
  public Account selectByUsername(String username) {
    return accountRepository.selectByUsername(username);
  }
}
