package com.supwisdom.institute.backend.base.domain.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.supwisdom.institute.backend.base.domain.entity.Permission;
import com.supwisdom.institute.backend.base.domain.model.PermissionTreeNode;
import com.supwisdom.institute.backend.base.domain.repo.PermissionRepository;
import com.supwisdom.institute.backend.common.framework.entity.EntityUtils;
import com.supwisdom.institute.backend.common.framework.service.ABaseService;

@Slf4j
@Service
public class PermissionService extends ABaseService<Permission, PermissionRepository> {
  
  @Override
  public PermissionRepository getRepo() {
    return permissionRepository;
  }

  @Autowired
  private PermissionRepository permissionRepository;
  
  @Override
  public Permission insert(Permission entity) {
    Permission parentPermission = permissionRepository.selectById(entity.getParentId());
    if (parentPermission != null) {
      entity.setApplicationId(parentPermission.getApplicationId());
    }
    
    return super.insert(entity);
  }

  public void deleteBatch(List<String> ids) {
    
    ids.stream().forEach(id -> {
      this.deleteById(id);
    });
  }
  
  private Permission selectApplicationPermission() {
    Permission permission = permissionRepository.selectById(Permission.APPLICATION_ID);
    
    return permission;
  }
  
  private List<Permission> selectApplicationPermissionList() {
    Map<String, Object> mapBean = new HashMap<String, Object>();
    mapBean.put("type", Permission.TYPE_APPLICATION);
    
    return permissionRepository.selectList(true, -1, -1, mapBean, null);
  }

  public PermissionTreeNode selectPermissionTree(String applicationId, String type) {

    Map<String, Object> mapBean = new HashMap<String, Object>();
    if (StringUtils.isNotBlank(type)) {
      mapBean.put("type", type);
    }
    
    String rootTreeNodeId = "0";
    
    List<Permission> permissions = new ArrayList<Permission>();
    
    if (StringUtils.isBlank(applicationId)) {
      Permission rootPermission = new Permission();
      rootPermission.setId(Permission.ROOT_PARENT_ID);
      rootPermission.setCode("root");
      rootPermission.setName("根");
      
      permissions.add(rootPermission);
      rootTreeNodeId = rootPermission.getId();

      List<Permission> applicationPermissions = this.selectApplicationPermissionList();
      permissions.addAll(applicationPermissions);
    } else {
      Permission applicationPermission = permissionRepository.selectById(applicationId); // this.selectApplicationPermissionByCode(applicationCode);
      if (applicationPermission == null) {
        return null;
      }
      
      permissions.add(applicationPermission);
      rootTreeNodeId = applicationPermission.getId();
    }
    
    List<Permission> menuPermissions = permissionRepository.selectList(true, -1, -1, mapBean, null);
    permissions.addAll(menuPermissions);

    return convertPermissionTree(permissions, rootTreeNodeId);
  }
  
  private PermissionTreeNode convertPermissionTree(List<Permission> permissions, String rootTreeNodeId) {
    if (permissions == null || permissions.size() == 0) {
      return null;
    }

    Map<String, PermissionTreeNode> parentTreeNode = new LinkedHashMap<String, PermissionTreeNode>();

    for (Permission permission : permissions) {
      PermissionTreeNode treeNode = EntityUtils.copy(permission, new PermissionTreeNode());
      treeNode.setChildren(new ArrayList<PermissionTreeNode>());
      
      if (parentTreeNode.containsKey(treeNode.getId())) {
        continue;
      }
      parentTreeNode.put(treeNode.getId(), treeNode);
    }
    
    for (PermissionTreeNode treeNode : parentTreeNode.values()) {
      if (!parentTreeNode.containsKey(treeNode.getParentId())) {
        continue;
      }
      
      parentTreeNode.get(treeNode.getParentId()).getChildren().add(treeNode);
    }
    
    return parentTreeNode.get(rootTreeNodeId);
  }

  

//  public Permission selectApplicationPermissionByCode(String code) {
//    Permission probe = new Permission();
//    probe.setCode(code);
//    probe.setType("1");
//    
//    ExampleMatcher matcher = ExampleMatcher.matching()
//        .withMatcher("code", ExampleMatcher.GenericPropertyMatchers.exact())
//        .withMatcher("type", ExampleMatcher.GenericPropertyMatchers.exact());
//    
//    Example<Permission> example = Example.of(probe, matcher);
//    
//    Optional<Permission> o = permissionRepository.findOne(example);
//    
//    if (o.isPresent()) {
//      return o.get();
//    }
//    
//    return null;
//  }
  

  public List<Permission> selectByUsername(String username, String applicationId, String type) {
    List<Permission> permissions = new ArrayList<Permission>();
    
//    Permission applicationPermission = permissionRepository.selectById(applicationId);
//    if (applicationPermission == null) {
//      return permissions;
//    }
//    
//    int lft = applicationPermission.getLft();
//    int rgt = applicationPermission.getRgt();
    
    if (StringUtils.isBlank(applicationId)) {
      List<Permission> accountRolePermissions = permissionRepository.selectAccountRolePermissionByUsername(username, type);
      permissions.addAll(accountRolePermissions);
      
      List<Permission> accountGroupRolePermissions = permissionRepository.selectAccountGroupRolePermissionByUsername(username, type);
      permissions.addAll(accountGroupRolePermissions);
    } else {
      List<Permission> accountRolePermissions = permissionRepository.selectAccountRolePermissionByUsername(username, applicationId, type);
      permissions.addAll(accountRolePermissions);
      
      List<Permission> accountGroupRolePermissions = permissionRepository.selectAccountGroupRolePermissionByUsername(username, applicationId, type);
      permissions.addAll(accountGroupRolePermissions);
    }
    
    return EntityUtils.distinctList(permissions);
  }
  
  @Deprecated
  public PermissionTreeNode selectPermissionTreeByUsername(String username, String applicationId, String type) {
    String rootTreeNodeId = "0";
    
    List<Permission> permissions = new ArrayList<Permission>();
    
    if (StringUtils.isBlank(applicationId)) {
      Permission rootPermission = new Permission();
      rootPermission.setId(Permission.ROOT_PARENT_ID);
      rootPermission.setCode("root");
      rootPermission.setName("根");
      
      permissions.add(rootPermission);
      rootTreeNodeId = rootPermission.getId();

      List<Permission> applicationPermissions = this.selectApplicationPermissionList();
      permissions.addAll(applicationPermissions);
    } else {
      Permission applicationPermission = permissionRepository.selectById(applicationId); // this.selectApplicationPermissionByCode(applicationCode);
      if (applicationPermission == null) {
        return null;
      }
      
      permissions.add(applicationPermission);
      rootTreeNodeId = applicationPermission.getId();
    }
    
    List<Permission> menuPermissions = this.selectByUsername(username, applicationId, type);
    permissions.addAll(menuPermissions);

    return convertPermissionTree(permissions, rootTreeNodeId);
  }

}
